import React from 'react';
import {observer} from 'mobx-react';
import {Table, TableColumnConfig} from '@yandex-data-ui/cloud-components';
import classNames from 'classnames';

import {NTentaclesApi} from '../../../api/tentacles/stubs/tentacles';
import {Panel} from '../../../components/Panel/Panel';
import {
    ALLOCATOR_SCHEDULING_HELP,
    ALLOCATOR_UNUSED_NODES_HELP,
    AVAILABILITY_NOT_REACHABLE_HELP,
    JUGGLER_NOT_PRESENT_HELP,
    JUGGLER_NOT_VISIBLE_HELP,
    WALLE_DECOMMISSIONED_HELP,
    WALLE_FREE_OR_PROBATION_HELP,
} from '../Overview/constants';

import styles from '../Sessions.module.css';
import stylesTable from '../../../design/patches/Table.module.css';
import padding from '../../../design/padding.module.css';

type Columns = keyof NTentaclesApi.TTentacles;

interface PipelinePodsViewProps {
    pods?: NTentaclesApi.TTentacles[];
    filter?: NTentaclesApi.EPodFilter;
    columns?: Columns[];
    total: number;
    className?: string;
}

const helpers: {[key in NTentaclesApi.EPodFilter]?: string} = {
    [NTentaclesApi.EPodFilter.EXCLUDED_DECOMMISSIONED_NODES]: WALLE_DECOMMISSIONED_HELP,
    [NTentaclesApi.EPodFilter.EXCLUDED_PROBATION_OR_FREE_NODES]: WALLE_FREE_OR_PROBATION_HELP,
    [NTentaclesApi.EPodFilter.EXCLUDED_NOT_VISIBLE]: JUGGLER_NOT_VISIBLE_HELP,
    [NTentaclesApi.EPodFilter.EXCLUDED_NOT_PRESENT]: JUGGLER_NOT_PRESENT_HELP,
    [NTentaclesApi.EPodFilter.PODS_SCHEDULING_ERRORS]: ALLOCATOR_SCHEDULING_HELP,
    [NTentaclesApi.EPodFilter.UNUSED_YP_NODES]: ALLOCATOR_UNUSED_NODES_HELP,
    [NTentaclesApi.EPodFilter.EXCLUDED_NOT_REACHABLE]: AVAILABILITY_NOT_REACHABLE_HELP,
};

const renderer: {[key in Columns]?: TableColumnConfig<NTentaclesApi.TTentacles>} = {
    walle_project: {
        id: 'walle_project',
        name: 'Walle project',
        template: (item) => (
            <a
                className={'link link_theme_normal'}
                href={`https://wall-e.yandex-team.ru/projects/hosts?project=${encodeURIComponent(item.walle_project)}`}
            >
                {item.walle_project}
            </a>
        ),
    },
    fqdn: {
        id: 'fqdn',
        name: 'Fqdn',
        template: (item) => (
            <a className={'link link_theme_normal'} href={`https://wall-e.yandex-team.ru/host/${item.fqdn}`}>
                {item.fqdn}
            </a>
        ),
    },
    walle_scenario_id: {
        id: 'walle_scenario_id',
        name: 'Walle scenario id',
        template: (item) =>
            !item.walle_scenario_id ? undefined : (
                <a
                    className={'link link_theme_normal'}
                    href={`https://wall-e.yandex-team.ru/scenarios/${item.walle_scenario_id}`}
                >
                    {item.walle_scenario_id}
                </a>
            ),
    },
    walle_ticket: {
        id: 'walle_ticket',
        name: 'Walle ticket',
        template: (item) =>
            !item.walle_ticket ? undefined : (
                <a className={'link link_theme_normal'} href={`https://st.yandex-team.ru/${item.walle_ticket}`}>
                    {item.walle_ticket}
                </a>
            ),
    },
};

export const PipelinePodsView: React.FunctionComponent<PipelinePodsViewProps> = observer((props) => {
    const columns2 = props.columns?.map((item) =>
        renderer.hasOwnProperty(item)
            ? renderer[item]!
            : {
                  id: item,
                  name: item,
              }
    ) || [
        {
            id: 'fqdn',
            name: 'Fqdn',
        },
    ];

    const help = props.filter ? helpers[props.filter] : undefined;

    return (
        <div className={classNames(styles.PodsTable, props.className)}>
            {help && (
                <Panel className={padding.topM}>
                    <div dangerouslySetInnerHTML={{__html: help}} />
                </Panel>
            )}
            <Table
                data={props.pods || []}
                columns={columns2}
                emptyMessage={props.filter !== undefined ? 'No data' : 'Pods filter is not selected'}
                className={classNames(stylesTable.fullsize)}
            />
        </div>
    );
});
