import React from 'react';
import {PipelineFiltersView} from './PipelineFiltersView';
import styles from '../Sessions.module.css';
import {Label, Loader} from '@yandex-data-ui/common';
import {PipelinePodsView} from './PipelinePodsView';
import {Columns, Pipeline} from '../../../stores/AllocationZoneStore';
import {NTentaclesApi} from '../../../api/tentacles/stubs/tentacles';
import {observer} from 'mobx-react';
import {AllocationZonePodsStore} from '../../../stores/AllocaitonZonePodsStore';
import {useSearchParam} from '../../../hooks/useSearchParam';
import {useStorePolling} from '../../../hooks/useStorePolling';

interface PipelineViewProps {
    pipeline: Pipeline;
    slices: Map<NTentaclesApi.EPodFilter, Columns[]>;
    zoneId: string;
}

export const PipelineView: React.FunctionComponent<PipelineViewProps> = observer((props) => {
    const [filterName, setFilterName] = useSearchParam('podsFilter');
    const setFilter = React.useCallback(
        (filter?: NTentaclesApi.EPodFilter) => {
            setFilterName(filter ? NTentaclesApi.EPodFilter[filter] : undefined);
        },
        [setFilterName]
    );
    // @ts-ignore: Unreachable code error
    const filter = NTentaclesApi.EPodFilter[filterName];

    const [store, setStore] = React.useState(
        filter ? new AllocationZonePodsStore(props.zoneId, props.slices.get(filter)!, filter) : undefined
    );

    React.useEffect(() => {
        if (filter === store?.filter) {
            return;
        }

        if (!filter) {
            setStore(undefined);
            return;
        }

        setStore(new AllocationZonePodsStore(props.zoneId, props.slices.get(filter)!, filter));
    }, [store, filter, props.zoneId, props.slices]);

    useStorePolling(store);

    return (
        <div style={{display: 'flex'}}>
            <PipelineFiltersView
                pipeline={props.pipeline}
                setFilter={setFilter}
                className={styles.PipelineList}
                filter={filter}
            />

            {store?.loading && !store.pods?.length ? (
                <div style={{display: 'flex', justifyContent: 'center', width: '100%', padding: '20px'}}>
                    <Loader />
                </div>
            ) : store?.error ? (
                <Label theme={'danger'} className={styles.LabelPanel} onClick={() => store?.load()}>
                    Request failed, click to refresh
                </Label>
            ) : (
                <PipelinePodsView
                    pods={store?.pods?.slice()}
                    filter={filter}
                    total={store?.total!}
                    columns={filter ? props.slices.get(filter) : []}
                />
            )}
        </div>
    );
});
