import React from 'react';
import {Datepicker, Dialog, TimelineProps, YCSelect} from '@yandex-data-ui/common';
import {NTentaclesApi} from '../../api/tentacles/stubs/tentacles';
import {Button, TextInput} from 'lego-on-react';
import * as yup from 'yup';
import {useFormik} from 'formik';
import {DatepickerOutputDates} from '@yandex-data-ui/common/build/components/Datepicker/Datepicker';
import {IncidentFiltersContext} from '../../context/IncidentFiltersContext';
import {observer} from 'mobx-react';
import {createIncident} from '../../api/tentacles/creteIncident';
import {updateIncident} from '../../api/tentacles/updateIncident';
import {startOfDay, startOfMinute} from 'date-fns';
import {fetchIncidentData} from '../../api/tentacles/fetchIncidentData';

const {ESloType} = NTentaclesApi;

interface CreateIncidentProps {
    onClose: () => void;
    onDelete: (id: string) => void;
    initialValues: NTentaclesApi.ITIncident;
}

const schema = yup.object({
    allocation_zone: yup.string().label('Allocation zone'),
    slo_type: yup.number().label('Slo type'),
    assignee: yup.string().label('Assignee'),
    minutes_out_of_slo: yup.number().typeError('not a number').label('Out of SLO'),
    start_time_ts: yup.number().label('Start time'),
    end_time_ts: yup.number().label('End time'),
});

function useFormikInput<T>(setFieldValue: (name: keyof T, value: any) => void) {
    type Key = Extract<keyof T, string>;
    // type DoubleKey = `${Key}-${Key}`;
    type TextHandler = {[key in Key]: (value: string) => void};
    type DatepickerFirstHandler = {[key in Key]: DatepickerSecondHandler};
    type DatepickerSecondHandler = {[key in Key]: (value: DatepickerOutputDates) => void};
    type TimelineFirstHandler = {[key in Key]: TimelineSecondHandler};
    type TimelineSecondHandler = {[key in Key]: TimelineProps['onChange']};

    return React.useMemo(
        () => ({
            text: new Proxy<TextHandler>({} as any, {
                get(_, p: Key) {
                    return (value: string) => {
                        setFieldValue(p, value);
                    };
                },
            }),
            enum<T extends {[key in string | number]: any}>(values: T) {
                type EnumHandler = {[key in Key]: (value: string | number) => void};
                return new Proxy<EnumHandler>({} as any, {
                    get(_, p: Key) {
                        return (value: string | number) => {
                            setFieldValue(p, values[value]);
                        };
                    },
                });
            },
            number: new Proxy<TextHandler>({} as any, {
                get(_, p: Key) {
                    return (value: string) => {
                        setFieldValue(p, Number(value));
                    };
                },
            }),
            datepicker: new Proxy<DatepickerFirstHandler>({} as any, {
                get(_, p1: Key) {
                    return new Proxy<DatepickerSecondHandler>({} as any, {
                        get(_, p2: Key) {
                            return (value: DatepickerOutputDates) => {
                                const from = Date.parse(value.from || '');
                                const to = Date.parse(value.to || '');

                                setFieldValue(p1, Math.floor(from / 1000));
                                setFieldValue(p2, Math.floor(to / 1000));
                            };
                        },
                    });
                },
            }),
            timeline: new Proxy<TimelineFirstHandler>({} as any, {
                get(_, p1: Key) {
                    return new Proxy<TimelineSecondHandler>({} as any, {
                        get(_, p2: Key): TimelineProps['onChange'] {
                            return ({from, to}) => {
                                // const from = Date.parse(value.from || '');
                                // const to = Date.parse(value.to || '');

                                setFieldValue(p1, Math.floor(from / 1000));
                                setFieldValue(p2, Math.floor(to / 1000));
                            };
                        },
                    });
                },
            }),
        }),
        [setFieldValue]
    );
}

export const CreateIncidentModal: React.FunctionComponent<CreateIncidentProps> = observer((props) => {
    const filters = React.useContext(IncidentFiltersContext);
    const [initialValues, setInitialValues] = React.useState(props.initialValues);
    const [error, setError] = React.useState<string>();
    const [loading, setLoading] = React.useState<boolean>(false);

    const reloadIncident = React.useCallback(
        async (id: string) => {
            const response = await fetchIncidentData({id});
            const result = NTentaclesApi.TReadIncidentDataResult.create(response.result!);
            setInitialValues(result.data || {});
        },
        [setInitialValues]
    );

    const onSubmit = React.useCallback(
        async (data: NTentaclesApi.ITIncident) => {
            const upsert = !initialValues.id ? createIncident : updateIncident;
            setLoading(true);
            try {
                const response = await upsert({data});
                // @ts-ignore: use result.id from TCreateIncidentResponse
                const incidentId = response.result?.id;
                if (incidentId) {
                    await reloadIncident(incidentId);
                }
            } catch (e) {
                setError(String(e));
            } finally {
                setLoading(false);
            }
        },
        [initialValues, setError, setLoading, reloadIncident]
    );

    const formik = useFormik<NTentaclesApi.ITIncident>({
        initialValues: initialValues || {
            start_time_ts: startOfDay(Date.now()).getTime() / 1000,
            end_time_ts: startOfMinute(Date.now()).getTime() / 1000,
        },
        validationSchema: schema,
        onSubmit,
    });

    React.useEffect(() => {
        if (!initialValues) {
            return;
        }

        if (initialValues.allocation_zone && !filters.allocation_zones.includes(initialValues.allocation_zone)) {
            filters.allocation_zones.push(initialValues.allocation_zone);
        }

        if (initialValues.assignee && !filters.assignees.includes(initialValues.assignee)) {
            filters.assignees.push(initialValues.assignee);
        }
    }, [initialValues, filters]);

    React.useEffect(() => {
        formik.setValues(initialValues);
    }, [initialValues, formik.setValues]);

    const onInput = useFormikInput<NTentaclesApi.ITIncident>(formik.setFieldValue);

    return (
        <Dialog visible={true} onClose={props.onClose} hasButtonClose={false} autoclosable={false}>
            <Dialog.Body>
                <h3>
                    {initialValues.id ? (
                        <>
                            Update Incident{' '}
                            <span style={{color: 'var(--yc-color-text-secondary)'}}>#{initialValues.id}</span>
                        </>
                    ) : (
                        'New Incident'
                    )}
                </h3>
                <div style={{display: 'grid', gridTemplateColumns: 'auto 300px', alignItems: 'center', gap: '16px'}}>
                    <label htmlFor={'allocation_zone'}>Allocation Zone</label>
                    <div>
                        <YCSelect<'single'>
                            onChange={onInput.text.allocation_zone}
                            items={filters.allocation_zones.map((item) => ({
                                value: item,
                                title: item,
                            }))}
                            addItem={filters.allocation_zones.push.bind(filters.allocation_zones)}
                            allowNullableValues={true}
                            showSearch={true}
                            value={formik.values.allocation_zone!}
                            placeholder={'Allocation zone'}
                        />
                        {formik.errors.allocation_zone}
                    </div>
                    <label htmlFor={'assignee'}>Assignee</label>
                    <div>
                        <YCSelect<'single'>
                            onChange={onInput.text.assignee}
                            items={filters.assignees.map((item) => ({
                                value: item,
                                title: item,
                            }))}
                            addItem={filters.assignees.push.bind(filters.assignees)}
                            allowNullableValues={true}
                            showSearch={true}
                            value={formik.values.assignee!}
                            placeholder={'Assignee'}
                        />
                        {formik.errors.assignee}
                    </div>
                    <label htmlFor={'ticket'}>Ticket</label>
                    <div>
                        <TextInput
                            text={formik.values.startrek_ticket_id!}
                            onChange={onInput.text.startrek_ticket_id}
                            placeholder={'Ticket'}
                            id={'ticket'}
                            theme={'normal'}
                            view={'default'}
                            size={'s'}
                            tone={'default'}
                        />
                    </div>
                    <label htmlFor={'out_of_slo'}>Out of SLO, min</label>
                    <div>
                        <div style={{width: 200}}>
                            <TextInput
                                text={formik.values.minutes_out_of_slo!}
                                onChange={onInput.text.minutes_out_of_slo}
                                placeholder={'Out of SLO'}
                                id={'out_of_slo'}
                                theme={'normal'}
                                view={'default'}
                                size={'s'}
                                tone={'default'}
                            />
                            {formik.errors.minutes_out_of_slo}
                        </div>
                    </div>
                    <label htmlFor={'slo_type'}>SLO Type</label>
                    <div style={{width: 200}}>
                        <YCSelect<'single'>
                            onChange={onInput.enum(NTentaclesApi.ESloType).slo_type}
                            items={[
                                {
                                    value: ESloType[ESloType.AVAILABILITY],
                                    title: ESloType[ESloType.AVAILABILITY],
                                },
                                {
                                    value: ESloType[ESloType.REALLOCATION],
                                    title: ESloType[ESloType.REALLOCATION],
                                },
                                {
                                    value: ESloType[ESloType.REDEPLOYED_ON_TIME],
                                    title: ESloType[ESloType.REDEPLOYED_ON_TIME],
                                },
                            ]}
                            allowNullableValues={true}
                            showSearch={false}
                            value={NTentaclesApi.ESloType[formik.values.slo_type!]}
                            placeholder={'SLO type'}
                        />
                        {formik.errors.slo_type}
                    </div>
                    <label>Time</label>
                    <div>
                        <Datepicker
                            format={'dd.MM.yyyy HH:mm'}
                            from={formik.values.start_time_ts ? formik.values.start_time_ts * 1000 : null}
                            to={formik.values.end_time_ts ? formik.values.end_time_ts * 1000 : null}
                            onChange={onInput.datepicker.start_time_ts.end_time_ts}
                        />
                    </div>
                </div>
            </Dialog.Body>
            <Dialog.Footer
                preset="default"
                onClickButtonCancel={props.onClose}
                onClickButtonApply={() => formik.handleSubmit()}
                textButtonApply="Save"
                textButtonCancel="Cancel"
                progress={loading}
                errorText={error}
                showError={Boolean(error)}
                listenKeyEnter
                renderButtons={(buttonApply, buttonCancel) => (
                    <div style={{display: 'flex'}}>
                        {initialValues.id && (
                            <Button
                                size={'m'}
                                theme={'normal'}
                                view={'default'}
                                tone={'default'}
                                text={'Delete'}
                                onClick={() => props.onDelete(initialValues.id!)}
                            />
                        )}
                        {buttonCancel}
                        {buttonApply}
                    </div>
                )}
            />
        </Dialog>
    );
});
