import React from 'react';
import {observer} from 'mobx-react';
import classNames from 'classnames';

import {IncidentsStore} from '../../stores/IncidentsStore';
import {Datepicker, DatepickerProps, YCSelect} from '@yandex-data-ui/common';
import {Table, withTableActions} from '@yandex-data-ui/cloud-components';
import {NTentaclesApi} from '../../api/tentacles/stubs/tentacles';
import {CreateIncidentModal} from './CreateIncidentModal';
import {Button} from 'lego-on-react';
import {addDays, endOfDay, format, formatDuration, intervalToDuration, startOfDay} from 'date-fns';
import {Panel} from '../../components/Panel/Panel';

import styles from './Incidents.module.css';
import {DeleteIncidentModal} from './DeleteIncidentModal';
import {IncidentFiltersContext} from '../../context/IncidentFiltersContext';
import {useDoctitle} from '../../hooks/useDoctitle';

import flex from '../../design/flex.module.css';
import padding from '../../design/padding.module.css';
import divider from '../../design/divider.module.css';
import typography from '../../design/typography.module.css';

interface IncidentsListProps {
}

const TableWithActions = withTableActions(Table);

export const IncidentsList: React.FunctionComponent<IncidentsListProps> = observer((props) => {
    const now = endOfDay(Date.now()).getTime();
    const filters = React.useContext(IncidentFiltersContext);
    const [store] = React.useState(new IncidentsStore(startOfDay(addDays(now, -365)).getTime(), now));
    const [upsertIncident, setUpsertIncident] = React.useState<undefined | NTentaclesApi.ITIncident>();
    const [deleteIncidentId, setDeleteIncidentId] = React.useState<string>();

    const onDateChange: DatepickerProps['onChange'] = React.useCallback(
        ({from, to}) => {
            store.updateFilter({
                started_after_ts: from ? startOfDay(Date.parse(from)).getTime() : undefined,
                ended_before_ts: to ? endOfDay(Date.parse(to)).getTime() : undefined,
            });

            store.load();
        },
        [store],
    );

    React.useEffect(() => {
        store.load();
    }, [store]);

    useDoctitle('Incidents');

    return (
        <div>
            <Panel className={classNames(flex.container, flex.center, divider.bottom)}>
                <div className={classNames(flex.grow1, typography.body3)}>Incidents</div>

                <Button
                    theme={'action'}
                    view={'default'}
                    size={'m'}
                    tone={'default'}
                    title={'New Incident'}
                    text={'New Incident'}
                    onClick={() => setUpsertIncident({})}
                />
            </Panel>
            <Panel className={classNames(padding.topL)}>
                <div style={{display: 'flex'}}>
                    <div style={{flexGrow: 1, display: 'grid', gap: 20, gridTemplateColumns: 'auto 200px 200px 1fr'}}>
                        <Datepicker
                            onChange={onDateChange}
                            from={store.filters.started_after_ts ? store.filters.started_after_ts * 1000 : null}
                            to={store.filters.ended_before_ts ? store.filters.ended_before_ts * 1000 : null}
                            allowNullableValues={true}
                        />

                        <YCSelect<'multiple'>
                            type={YCSelect.MULTIPLE}
                            onChange={(items) => {
                                store.updateFilter({
                                    allocation_zones: items,
                                });
                                store.load();
                            }}
                            items={filters.allocation_zones.map((item) => ({
                                value: item,
                                title: item,
                            }))}
                            addItem={(item) => {
                                // @ts-ignore: incorrect type on callback
                                filters.allocation_zones.push(item);
                            }}
                            allowNullableValues={true}
                            showSearch={true}
                            value={store.filters.allocation_zones || []}
                            placeholder={'Allocation zone'}
                        />

                        <YCSelect<'multiple'>
                            type={YCSelect.MULTIPLE}
                            onChange={(items) => {
                                store.updateFilter({
                                    assignees: items,
                                });
                                store.load();
                            }}
                            items={filters.assignees.map((item) => ({
                                value: item,
                                title: item,
                            }))}
                            addItem={(item) => {
                                // @ts-ignore: incorrect type on callback
                                filters.assignees.push(item as string);
                            }}
                            allowNullableValues={true}
                            showSearch={true}
                            value={store.filters.assignees || []}
                            placeholder={'Assignees'}
                        />
                    </div>
                </div>
            </Panel>
            <div style={{width: '100%'}}>
                {upsertIncident !== undefined && (
                    <CreateIncidentModal
                        initialValues={upsertIncident}
                        onClose={() => {
                            setUpsertIncident(undefined);
                            store.load();
                        }}
                        onDelete={setDeleteIncidentId}
                    />
                )}
                {deleteIncidentId && (
                    <DeleteIncidentModal
                        id={deleteIncidentId}
                        onClose={() => {
                            setDeleteIncidentId(undefined);
                        }}
                        onDelete={() => {
                            setUpsertIncident(undefined);
                            setDeleteIncidentId(undefined);
                            store.load();
                        }}
                    />
                )}
                <TableWithActions
                    className={styles.FullSizeTableWrapper}
                    data={store.incidents}
                    getRowActions={(item) => [
                        {
                            text: 'Edit',
                            handler: () => {
                                setUpsertIncident(item as NTentaclesApi.TIncident);
                            },
                        },
                        {
                            text: 'Clone',
                            handler: () => {
                                const {id, ...clone} = item;
                                setUpsertIncident(clone);
                            },
                        },
                        {
                            text: 'Delete',
                            handler: () => {
                                setDeleteIncidentId(item.id);
                            },
                            theme: 'danger',
                        },
                    ]}
                    columns={[
                        {
                            id: 'overview',
                            name: '',
                            template(item) {
                                return (
                                    <div>
                                        <div>{item.minutes_out_of_slo} min out of SLO</div>
                                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                                            #{item.id}
                                            <br />
                                            Incident started at {format(item.start_time_ts * 1000, 'yyyy.MM.dd HH:mm')}
                                            <br />
                                            Duration is{' '}
                                            {formatDuration(
                                                intervalToDuration({
                                                    start: item.start_time_ts * 1000,
                                                    end: item.end_time_ts * 1000,
                                                }),
                                            )}
                                        </div>
                                    </div>
                                );
                            },
                        },
                        {
                            id: 'allocation_zone',
                            name: 'Allocation Zone',
                        },
                        {
                            id: 'assignee',
                            name: 'Assignee',
                        },
                        {
                            id: 'startrek_ticket_id',
                            name: 'Ticket',
                            template: (item, defaultValue) =>
                                Boolean(item.startrek_ticket_id) ? (
                                    <a
                                        className={'link link_theme_normal'}
                                        href={`https://st.yandex-team.ru/${item.startrek_ticket_id}`}
                                    >
                                        {item.startrek_ticket_id}
                                    </a>
                                ) : undefined,
                        },
                        {
                            id: 'slo_type',
                            name: 'Slo type',
                            template: (item) => NTentaclesApi.ESloType[item.slo_type],
                        },
                    ]}
                    emptyMessage={'No data'}
                />
            </div>
        </div>
    );
});
