import React from 'react';
import {observer} from 'mobx-react';
import {of} from 'rxjs';
import {delay, mergeMap, repeat} from 'rxjs/operators';
import {MissingNodesFilter, MissingNodesStore} from '../../stores/MissingNodesStore';
import {Table} from '@yandex-data-ui/cloud-components';
import {TableColumnConfig} from '@yandex-data-ui/cloud-components/build/components/Table/Table';
import {Link, useLocation} from 'react-router-dom';
import routes from '../../routes';
import {Button} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {Panel} from '../../components/Panel/Panel';

import stylesTable from '../../design/patches/Table.module.css';
import {useDoctitle} from '../../hooks/useDoctitle';

interface MissingNodesProps {}

export const MissingNodes: React.FunctionComponent<MissingNodesProps> = observer(() => {
    useDoctitle('Missing nodes');
    const {search} = useLocation();

    const searchFilter = React.useMemo<MissingNodesFilter | undefined>(() => {
        const params = new URLSearchParams(search);
        const filter = params.get('filter');

        if (!filter) {
            return undefined;
        }

        const [name, value] = decodeURIComponent(filter).split(':');
        return {name, value};
    }, [search]);

    const searchLimit = React.useMemo<number>(() => {
        const params = new URLSearchParams(search);
        const limit = params.get('limit');
        return limit ? Number(limit) : 200;
    }, [search]);

    const [store] = React.useState(new MissingNodesStore(searchFilter, searchLimit));
    const loadMore = React.useCallback(() => {
        store.loadMore();
    }, [store]);

    React.useEffect(() => {
        store.setFilter(searchFilter);
        store.setLimit(searchLimit);
    }, [store, searchFilter, searchLimit]);

    const columns = React.useMemo(() => store.columns.slice(), [store.columnsHash]);
    const filter = React.useMemo(() => (store.filter ? {...store.filter} : undefined), [store.filter]);

    const columnsConfig = React.useMemo<TableColumnConfig<Record<string, any>>[]>(() => {
        return columns.map((id) => ({
            id,
            template: (item) => {
                const value = item[id];
                if (id === 'count' || id === 'fqdn') {
                    return value;
                }

                if (filter?.name === id && filter?.value === value) {
                    return (
                        <Link
                            className={'link link_theme_normal'}
                            style={{textDecoration: 'underline'}}
                            to={routes.missingNodes}
                        >
                            {value}
                        </Link>
                    );
                }

                return (
                    <Link
                        to={`${routes.missingNodes}?filter=${encodeURIComponent(`${id}:${value}`)}`}
                        className={'link link_theme_normal'}
                    >
                        {value}
                    </Link>
                );
            },
        }));
    }, [columns, filter]);

    React.useEffect(() => {
        const polling = of({})
            .pipe(
                mergeMap(() => store.load()),
                delay(5000),
                repeat()
            )
            .subscribe();

        return () => {
            polling.unsubscribe();
        };
    }, [store]);

    return (
        <>
            {store.values.length === 0 && store.loading ? (
                <Panel style={{display: 'flex', justifyContent: 'center'}}>
                    <Loader />
                </Panel>
            ) : (
                <>
                    {' '}
                    <Table className={stylesTable.fullsize} data={store.values.slice()} columns={columnsConfig} />
                    {store.hasMore && (
                        <div style={{display: 'flex', justifyContent: 'center'}}>
                            <Button
                                size={'m'}
                                text={'Load more'}
                                onClick={loadMore}
                                theme={'normal'}
                                view={'default'}
                                tone={'default'}
                            />
                        </div>
                    )}
                </>
            )}
        </>
    );
});
