import React from 'react';
import {Label} from '@yandex-data-ui/common';
import dayjs from 'dayjs';

import {NTentaclesApi} from '../../api/tentacles/stubs/tentacles';
import {ZoneOverview} from '../../stores/SLOStore';

import styles from './SLOZoneFeature.module.css';
import {AllowedFeatures} from './SLOZone';

interface SLOZoneFeatureProps {
    featureName: AllowedFeatures;
    overview: ZoneOverview;
}

export const SLOZoneFeature: React.FunctionComponent<SLOZoneFeatureProps> = ({overview, featureName}) => {
    let label: string = featureName;
    let status = NTentaclesApi.EFeatureStatus[NTentaclesApi.EFeatureStatus.UNKNOWN];
    let description = '';

    if (featureName === 'dataFreshness') {
        const {dataFreshness} = overview;

        if (dataFreshness.status === NTentaclesApi.EFeatureStatus.OK) {
            return null;
        }

        label = 'freshness';
        description = `from ${dayjs(dataFreshness.data_timestamp * 1000).format('YYYY.MM.DD HH:mm')}`;
    }

    if (featureName === 'availability') {
        const {availability} = overview;
        status = `${Math.floor((1000 * availability.ok_count) / availability.total_count) / 10}%`;
        description = `of ${availability.total_count} is available`;
    }

    if (featureName === 'unusedNodes') {
        const {unusedNodes} = overview;

        if (!unusedNodes.enable) {
            return null;
        }

        label = 'used';
        status = `${
            Math.floor((1000 * (unusedNodes.total_count - unusedNodes.unused_count)) / unusedNodes.total_count) / 10
        }%`;
        description = `of ${unusedNodes.total_count} nodes`;
    }

    if (featureName === 'schedulingErrors') {
        const {schedulingErrors} = overview;

        if (!schedulingErrors.enable) {
            return null;
        }

        label = 'YP errors ';
        status = `${
            Math.floor(
                (1000 *
                    (schedulingErrors.scheduling_errors_on_up_nodes_count)) /
                    schedulingErrors.total_up_nodes_count
            ) / 10
        }%`;
        description = `on ${schedulingErrors.total_up_nodes_count} nodes`;
    }

    if (featureName === 'redeploy') {
        const {redeploy} = overview;
        status = `${Math.floor((1000 * redeploy.ok_count) / redeploy.total_count) / 10}%`;
        description = `of ${redeploy.total_count} is deployed`;
    }

    if (featureName === 'reallocation') {
        const {reallocation} = overview;

        if (!reallocation.enable) {
            return null;
        }

        const current = reallocation.current_reallocation_timestamp * 1000;
        const last = reallocation.last_success_reallocation_timestamp * 1000;

        if (current !== 0) {
            const delay = Date.now() - current;
            status = `${Math.floor(delay / (1000 * 60))} min`;
            description = 'passed after start';
        } else if (last !== 0) {
            const delay = Date.now() - last;
            status = `${Math.floor(delay / (1000 * 60))} min`;
            description = 'passed after finish';
        }
    }

    return (
        <React.Fragment key={featureName}>
            <span>{label}</span>
            <Label theme={overview[featureName].status === NTentaclesApi.EFeatureStatus.OK ? 'success' : 'danger'}>
                {status}
            </Label>
            <span className={styles.description}>{description}</span>
        </React.Fragment>
    );
};
