import React, {useEffect, useState} from 'react';
import {of} from 'rxjs';
import {catchError, delay, mergeMap, repeat, tap} from 'rxjs/operators';
import {toaster} from '@yandex-data-ui/common';
import {format} from 'date-fns';

export function usePolling(cb?: () => PromiseLike<any>) {
    const [lastUpdate, setLastUpdate] = useState(Date.now());
    const [error, setError] = useState<Error>();

    React.useEffect(() => {
        if (!cb) {
            return;
        }

        const polling = of({})
            .pipe(
                mergeMap(cb),
                tap(() => {
                    setError(undefined);
                    setLastUpdate(Date.now());
                }),
                catchError((err) => {
                    setError(err);
                    return of({});
                }),
                delay(5000),
                repeat()
            )
            .subscribe();

        return () => {
            polling.unsubscribe();
        };
    }, [cb, setError, setLastUpdate]);

    useEffect(() => {
        if (!error?.message) {
            return;
        }

        toaster?.createToast({
            name: 'network-error',
            title: error.message,
            allowAutoHiding: false,
            isClosable: false,
            type: 'error',
            content: 'Last success update: ' + format(lastUpdate, 'dd.MM.yyyy HH:mm'),
        });

        return () => {
            toaster?.removeToast('network-error');
        };
    }, [error?.message]);
}
