import {makeAutoObservable, observable, runInAction} from 'mobx';

import {NTentaclesApi} from '../api/tentacles/stubs/tentacles';
import {fetchPodsList} from '../api/tentacles/fetchPodsList';

type Columns = Extract<keyof NTentaclesApi.TTentacles, string>;

export class AllocationZonePodsStore {
    zoneId: string;
    pods?: NTentaclesApi.TTentacles[];
    total?: number;
    filter: NTentaclesApi.EPodFilter;
    columns: Columns[] = ['fqdn'];
    loading = false;
    error?: string;
    abortController = new AbortController();

    constructor(zoneId: string, columns: Columns[], filter: NTentaclesApi.EPodFilter) {
        this.zoneId = zoneId;
        this.filter = filter;
        this.columns = columns;

        makeAutoObservable(this, {
            pods: observable.shallow,
        });
    }

    async load() {
        runInAction(() => {
            this.loading = true;
        });

        try {
            const response = await fetchPodsList(
                {
                    allocation_zone_id: this.zoneId,
                    columns: this.columns,
                    filter: this.filter,
                },
                {signal: this.abortController.signal}
            );
            const result = NTentaclesApi.TPodsListResult.create(response.result!);
            const pods = result.tentacles.map(NTentaclesApi.TTentacles.create);
            runInAction(() => {
                this.error = '';
                this.pods = pods;
                this.total = Number(result.total_count!);
            });
        } catch (e) {
            runInAction(() => {
                this.pods = [];
                this.error = String(e);
            });
        }

        runInAction(() => {
            this.abortController = new AbortController();
            this.loading = false;
        });
    }
}
