import {makeAutoObservable, observable, runInAction} from 'mobx';

import {fetchAllocationZoneDetails} from '../api/tentacles/fetchAlloctionZoneDetails';
import {NTentaclesApi} from '../api/tentacles/stubs/tentacles';

export type SessionType = 'reallocation' | 'redeployment';

export class Features {
    availability: NTentaclesApi.TAvailabilityFeature;
    redeploy: NTentaclesApi.TRedeployFeature;
    reallocation: NTentaclesApi.TReallocationFeature;
    freshness: NTentaclesApi.TDataFreshnessFeature;
    unusedNodes: NTentaclesApi.TUnusedNodesFeature;

    constructor(features: NTentaclesApi.TAllocationZoneFeatures) {
        this.availability = NTentaclesApi.TAvailabilityFeature.create(features.availability!);
        this.redeploy = NTentaclesApi.TRedeployFeature.create(features.redeploy!);
        this.reallocation = NTentaclesApi.TReallocationFeature.create(features.reallocation!);
        this.freshness = NTentaclesApi.TDataFreshnessFeature.create(features.data_freshness!);
        this.unusedNodes = NTentaclesApi.TUnusedNodesFeature.create(features.unused_nodes!);
    }
}

export class Pipeline {
    walle: NTentaclesApi.TWallePipelineNode;
    hq: NTentaclesApi.THqPipelineNode;
    availability: NTentaclesApi.TAvailabilityPipelineNode;
    juggler: NTentaclesApi.TJugglerPipelineNode;
    redeployment: NTentaclesApi.TRedeploymentPipelineNode;
    allocator: NTentaclesApi.TAllocatorPipelineNode;

    constructor(pipeline: NTentaclesApi.ITPipeline) {
        this.walle = NTentaclesApi.TWallePipelineNode.create(pipeline.walle_node!);
        this.hq = NTentaclesApi.THqPipelineNode.create(pipeline.hq_node!);
        this.availability = NTentaclesApi.TAvailabilityPipelineNode.create(pipeline.availability_node!);
        this.juggler = NTentaclesApi.TJugglerPipelineNode.create(pipeline.juggler_node!);
        this.redeployment = NTentaclesApi.TRedeploymentPipelineNode.create(pipeline.redeployment_node!);
        this.allocator = NTentaclesApi.TAllocatorPipelineNode.create(pipeline.allocator_node!);
    }
}

export class Slices {
    constructor(pipeline: Pipeline) {
        const slices = [
            ...pipeline.walle.slices,
            ...pipeline.hq.slices,
            ...pipeline.availability.slices,
            ...pipeline.juggler.slices,
            ...pipeline.redeployment.slices,
            ...pipeline.allocator.slices,
        ];

        const slicesByFilter = new Map<NTentaclesApi.EPodFilter, string[]>();

        for (let item of slices) {
            const slice = NTentaclesApi.TSlice.create(item);
            slicesByFilter.set(slice.filter, slice.columns);
        }
    }
}

export class Freshness {
    hq: number;
    juggler: number;
    walle: number;
    yp: number;

    constructor(freshness: NTentaclesApi.ITFreshness) {
        const data = NTentaclesApi.TFreshness.create(freshness);
        this.hq = data.hq_ts * 1000;
        this.juggler = data.juggler_ts * 1000;
        this.walle = data.walle_ts * 1000;
        this.yp = data.yp_ts * 1000;
    }
}

export type Session = {type: SessionType; session: NTentaclesApi.TManageSession};
export type Columns = Extract<keyof NTentaclesApi.TTentacles, string>;

export class AllocationZoneStore {
    id: string;
    detailsLoading = false;
    detailsError = false;
    pipeline?: Pipeline;
    sessions: Session[] = [];
    redeploySessions: NTentaclesApi.TManageSession[] = [];
    reallocationSessions: NTentaclesApi.TManageSession[] = [];
    features?: Features;
    freshness?: Freshness;
    slices = new Map<NTentaclesApi.EPodFilter, Columns[]>();
    reallocation = false;
    abortController = new AbortController();

    constructor(zoneName: string) {
        this.id = zoneName;
        makeAutoObservable(this, {
            sessions: observable.shallow,
        });
    }

    async load() {
        runInAction(() => {
            this.detailsLoading = true;
        });
        try {
            const response = await fetchAllocationZoneDetails(this.id);
            const result = NTentaclesApi.TAllocationZoneDetailsResult.create(response.result!);
            runInAction(() => {
                this.pipeline = new Pipeline(result.pipeline!);
                const {reallocation, redeployment} = NTentaclesApi.TManageProcesses.create(result.manage_processes!);
                const features = NTentaclesApi.TAllocationZoneFeatures.create(result.features!);
                this.features = new Features(features);
                this.freshness = new Freshness(result.freshness!);
                const sessions: Session[] = [];
                if (reallocation?.sessions) {
                    sessions.push(
                        ...reallocation.sessions.map(NTentaclesApi.TManageSession.create).map((session) => ({
                            type: 'reallocation' as SessionType,
                            session,
                        }))
                    );
                }

                if (redeployment?.sessions) {
                    sessions.push(
                        ...redeployment.sessions.map(NTentaclesApi.TManageSession.create).map((session) => ({
                            type: 'redeployment' as SessionType,
                            session,
                        }))
                    );
                }

                if (this.slices.size === 0) {
                    const slices = [
                        ...this.pipeline.walle.slices,
                        ...this.pipeline.hq.slices,
                        ...this.pipeline.availability.slices,
                        ...this.pipeline.juggler.slices,
                        ...this.pipeline.redeployment.slices,
                        ...this.pipeline.allocator.slices,
                    ];

                    for (let item of slices) {
                        const slice = NTentaclesApi.TSlice.create(item);
                        this.slices.set(slice.filter, slice.columns as Columns[]);
                    }
                }

                this.sessions = sessions.sort((s1, s2) => s1.session.start_ts - s2.session.start_ts);
                this.detailsError = false;
            });
        } catch (e) {
            runInAction(() => {
                this.detailsError = true;
            });
        }

        runInAction(() => {
            this.detailsLoading = false;
        });
    }
}
