import {makeAutoObservable, observable, runInAction} from 'mobx';
import {fetchMissingNodesOverview} from '../api/tentacles/fetchMissingNodesOverview';
import {fetchMissingNodes} from '../api/tentacles/fetchMissingNodes';

export type MissingNodesFilter = { name: string; value: any };
const COLUMNS_WITH_FILTER_APPLIED = ['fqdn', 'project', 'operation_state', 'location_short_datacenter_name'];

export class MissingNodesStore {
    columns: string[] = [];
    columnsHash = '';
    values: { [key: string]: any }[] = [];
    filter?: MissingNodesFilter;
    limit: number;
    hasMore = false;
    loading = false;

    constructor(filter?: MissingNodesFilter, limit?: number) {
        this.filter = filter;
        this.limit = limit || 200;
        if (filter) {
            this.columns = COLUMNS_WITH_FILTER_APPLIED;
            this.columnsHash = this.columns.join();
        }
        makeAutoObservable(this, {
            values: observable.shallow,
        });
    }

    setFilter(filter?: MissingNodesFilter) {
        if (filter === this.filter) {
            return;
        }

        runInAction(async () => {
            this.filter = filter;
            this.columns = COLUMNS_WITH_FILTER_APPLIED;
            this.columnsHash = this.columns.join();
            this.limit = 200;
            this.values = [];
            this.hasMore = false;
            await this.load();
        });
    }

    setLimit(limit: number) {
        if (limit === this.limit) {
            return;
        }

        runInAction(() => {
            this.limit = limit;
            this.load();
        })
    }

    loadMore() {
        this.setLimit(this.limit + 200);
    }

    async load() {
        this.loading = true;
        const response = !this.filter ? await fetchMissingNodesOverview() : await fetchMissingNodes({
            columns: this.columns.join(','),
            limit: this.limit,
            [this.filter.name]: this.filter.value
        });

        runInAction(() => {
            const {values, columns} = response.result;
            this.columns = columns;
            this.columnsHash = this.columns.join();
            this.values = values.map((value) =>
                value.reduce(
                    (acc, item, i) => ({
                        ...acc,
                        [columns[i]]: item,
                    }),
                    {}
                )
            );
            this.hasMore = this.limit === values.length;
            this.loading = false;
        });
    }
}
