import {NTentaclesApi} from '../api/tentacles/stubs/tentacles';
import {fetchOverview} from '../api/tentacles/fetchOverview';
import {IObservableArray, makeObservable, observable, runInAction} from 'mobx';
import {NetworkError} from '../api/NetworkError';

export class ZoneOverview {
    zoneId: string;
    availability: NTentaclesApi.TAvailabilityFeature;
    dataFreshness: NTentaclesApi.TDataFreshnessFeature;
    redeploy: NTentaclesApi.TRedeployFeature;
    reallocation: NTentaclesApi.TReallocationFeature;
    unusedNodes: NTentaclesApi.TUnusedNodesFeature;
    schedulingErrors: NTentaclesApi.TSchedulingErrorsFeature;

    constructor(zoneFeatures: NTentaclesApi.TAllocationZoneFeatures) {
        this.zoneId = zoneFeatures.allocation_zone;
        this.availability = NTentaclesApi.TAvailabilityFeature.create(zoneFeatures.availability || {});
        this.dataFreshness = NTentaclesApi.TDataFreshnessFeature.create(zoneFeatures.data_freshness || {});
        this.redeploy = NTentaclesApi.TRedeployFeature.create(zoneFeatures.redeploy || {});
        this.reallocation = NTentaclesApi.TReallocationFeature.create(zoneFeatures.reallocation || {});
        this.unusedNodes = NTentaclesApi.TUnusedNodesFeature.create(zoneFeatures.unused_nodes || {});
        this.schedulingErrors = NTentaclesApi.TSchedulingErrorsFeature.create(zoneFeatures.scheduling_errors || {});
    }

    static create(zoneFeatures: NTentaclesApi.TAllocationZoneFeatures) {
        return new this(zoneFeatures);
    }
}

interface Overview {
    gencfg: IObservableArray<ZoneOverview>;
    ypRealocationEnabled: IObservableArray<ZoneOverview>;
    ypRealocationDisabled: IObservableArray<ZoneOverview>;
}

export class OverviewSLOStore {
    overview?: Overview;
    error?: Error;

    constructor() {
        makeObservable(this, {
            overview: observable,
            error: observable,
        });
    }

    read() {
        if (this.error) {
            throw this.error;
        }

        if (!this.overview) {
            throw this.load();
        }

        return this.overview;
    }

    async load() {
        try {
            const response = await fetchOverview();

            const result = NTentaclesApi.TDataOverviewResult.create(response.result!);
            const allZones = result.allocation_zones.map(NTentaclesApi.TAllocationZoneFeatures.create);

            runInAction(() => {
                let overview = this.overview;

                if (!overview) {
                    overview = observable.object({
                        gencfg: observable.array<ZoneOverview>(),
                        ypRealocationEnabled: observable.array<ZoneOverview>(),
                        ypRealocationDisabled: observable.array<ZoneOverview>(),
                        lastUpdate: Date.now(),
                    });
                    this.overview = overview;
                }

                overview.gencfg.clear();
                overview.ypRealocationEnabled.clear();
                overview.ypRealocationDisabled.clear();

                for (let zoneOverview of allZones) {
                    if (zoneOverview.tags?.deploy_engine === 'gencfg') {
                        overview.gencfg.push(new ZoneOverview(zoneOverview));
                    } else if (
                        zoneOverview.tags?.deploy_engine === 'yp_lite' &&
                        zoneOverview.reallocation?.enable === true
                    ) {
                        overview.ypRealocationEnabled.push(new ZoneOverview(zoneOverview));
                    } else if (
                        zoneOverview.tags?.deploy_engine === 'yp_lite' &&
                        zoneOverview.reallocation?.enable === false
                    ) {
                        overview.ypRealocationDisabled.push(new ZoneOverview(zoneOverview));
                    }
                }
            });
        } catch (e) {
            if (e instanceof NetworkError && this.overview) {
                throw e;
            }

            runInAction(() => {
                this.error = e;
            });
        }
    }
}
