"""
Library with functions for eventlog uploader script
"""

import yt.wrapper as yt

import argparse
import logging
import os
import subprocess
import time


DEFAULT_UPLOAD_PERIOD = 60


def get_parser():
    pars = argparse.ArgumentParser(description="Dump production eventlogs")
    pars.add_argument("--yt-proxy", help="Yt proxy")
    pars.add_argument("--backbone", default=False, action='store_true',
                      help="Use backbone instead fastbone network")
    pars.add_argument("--yt-token", help="Yt oauth token (for debug purpose only)")
    pars.add_argument("--yt-token-file", help="File with Yt oauth token")
    pars.add_argument("--yt-token-env", help="Environment variable with Yt oauth token")
    pars.add_argument("--evlogfolder", help="Output eventlog folder")
    pars.add_argument("--evlogdump", help="Path to evlogdump binary")
    pars.add_argument("--instance", help="Instance name")
    pars.add_argument("--skip-instance", type=str, default=None, help="Skip launch if instance have this substring")
    pars.add_argument("--log", help="Log path")
    pars.add_argument("--period", type=int, default=300, help="Period length in seconds")
    pars.add_argument("--sleep", type=int, default=3600, help="Sleep time (in seconds) between upload attempts")
    pars.add_argument("--chance", type=float, default=0.03,
                      help="Chance to be runned on every attempt. Should be in the interval (0, 1)")
    pars.add_argument("--days", type=int, default=4, help="Days to store")
    pars.add_argument("--max-batch", type=int, default=50000,
                      help="Max batch size for uploading to yt at one time")
    pars.add_argument("--enable-utf8", default=False, action='store_true',
                      help="Enable encode_utf8")
    return pars


class Uploader(object):
    def __init__(self, args):
        self.evlogfolder = args.evlogfolder
        self.evlogdump = args.evlogdump
        self.instance = args.instance
        self.log = args.log
        self.period = args.period
        self.days = args.days
        self.max_batch = args.max_batch
        self.encode_utf8 = args.enable_utf8
        self.init_yt(args)

    def init_yt(self, args):
        yt.config.set_proxy(args.yt_proxy)
        if not args.backbone:
            yt.config['proxy']['proxy_discovery_url'] = 'hosts/fb'
        if args.yt_token_file:
            with open(args.yt_token_file) as fp:
                yt_token = fp.read().strip()
        elif args.yt_token_env:
            yt_token = os.environ.get(args.yt_token_env)
        else:
            yt_token = args.yt_token

        assert yt_token
        yt.config.config['token'] = yt_token

    def upload(self, upload_period=DEFAULT_UPLOAD_PERIOD):
        last_ts = int(time.time())
        last_ts -= last_ts % upload_period
        current_ts = last_ts - self.period
        while current_ts < last_ts:
            ts1 = current_ts * 10**6
            current_ts += upload_period
            ts2 = current_ts * 10**6 - 1
            try:
                self.run_upload_process(ts1, ts2)
            except Exception as e:
                logging.exception("Got exception while trying to upload for {} and {}".format(ts1, ts2))
                if hasattr(e, 'response') and e.response is not None and e.response.text is not None:
                    logging.error("HTTP response text: {}".format(e.response.text))

    def run_upload_process(self, ts1, ts2):
        date_to_remove = time.strftime('%Y%m%d', time.gmtime(ts1 / 10.0**6 - self.days * 60 * 60 * 24))
        date_to_create = time.strftime('%Y%m%d', time.gmtime(ts1 / 10.0**6))
        date_to_check = time.strftime('%Y%m%d', time.gmtime(ts2 / 10.0**6))
        assert date_to_create == date_to_check
        remove_table_path = os.path.join(self.evlogfolder, date_to_remove)
        create_table_path = os.path.join(self.evlogfolder, date_to_create)
        self.prepare_tables(create_table_path, remove_table_path)

        evlogdump_cmd = [
            self.evlogdump,
            "-s", str(ts1),
            "-e", str(ts2),
            self.log
        ]
        logging.info("Running: {}".format(evlogdump_cmd))
        process = subprocess.Popen(
            evlogdump_cmd,
            stdout=subprocess.PIPE,
        )

        try:
            for records in self.chunk_records(process):
                self.upload_records_block(create_table_path, records)
        finally:
            return_code = process.poll()
            if return_code is None:
                process.kill()
                logging.critical('Evlogdump process didn\'t terminated normally and was killed')
            elif return_code != 0:
                logging.critical('Evlogdump returned non-zero exit code {}'.format(return_code))

    def prepare_tables(self, create_table_path, remove_table_path):
        # remove obsolete tables
        yt.remove(remove_table_path, force=True)

        # create new table
        channels = [['instance', 'ts', 'frame_id', 'event_type', 'event_data']]
        yt.create('table', create_table_path, attributes={"channels": channels}, recursive=True, ignore_existing=True)

    def chunk_records(self, process):
        records = []
        for line in process.stdout:
            fields = line.split('\t', 3)
            if len(fields) == 3:
                fields.append('')

            assert len(fields) == 4
            records.append({
                'instance': self.instance,
                'ts': int(fields[0]),
                'frame_id': int(fields[1]),
                'event_type': fields[2],
                'event_data': fields[3]
            })

            if len(records) >= self.max_batch:
                yield records
                records = []
        if records:
            yield records

    def upload_records_block(self, table_path, records):
        rec_num = len(records)
        logging.info("There are {} records to upload".format(rec_num))
        try:
            table = yt.TablePath(table_path, append=True)
            yt.write_table(table, records, format=yt.JsonFormat(attributes={"encode_utf8": self.encode_utf8}))
        except Exception:
            logging.error('Failed to upload {} records'.format(rec_num))
            raise
        else:
            logging.info("Uploaded {} records".format(rec_num))
