#include "controller.h"

#include <infra/service_controller/libs/endpoint_set_controller/endpoint_set_controller.h>

#include <infra/libs/controller/controller/error/whitelist_error.h>
#include <infra/libs/controller/controller/client_filter_helpers.h>

#include <infra/libs/sensors/macros.h>

#include <yt/yt/orm/library/filter_introspection.h>

#include <library/cpp/protobuf/yt/yt2proto.h>

#include <util/datetime/cputimer.h>

namespace NInfra::NServiceController {

namespace {

const TString SERVICE_CONTROLLER_SUPERVISOR = "service-controller";
const TSensorGroup EP_CTL_SENSOR_GROUP("endpoint_set_controller");

} // namespace

TString TEndpointSetManager::GetObjectId() const {
    TStringBuilder combinedId;
    for (auto it = EndpointSets_.begin(); it != EndpointSets_.end(); ++it) {
        if (it != EndpointSets_.begin()) {
            combinedId << ",";
        }
        combinedId << it->Meta().id();
    }
    return combinedId;
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TEndpointSetManager::GetDependentObjectsSelectArguments() const {
    NYP::NClient::TSelectObjectsOptions podsSelectOptions;
    podsSelectOptions.SetLimit(PodsSelectChunkSize_);

    auto endpointSetClientFilterConfig = ClientFilterConfig_;
    endpointSetClientFilterConfig.SetAdditionalFilter(EndpointSets_[0].Spec().pod_filter());
    return {
        {
            NYP::NClient::NApi::NProto::OT_POD
            , {
                "/meta/id"
                , "/status/agent/iss_summary"
                , "/status/agent/pod_agent_payload/status/ready/status"
                , "/status/ip6_address_allocations"
                , "/status/iss_conf_summaries"
                , "/spec/pod_agent_payload/spec/target_state"
            } /* selector */
            , "" /* filter */
            , podsSelectOptions /* options */
            , endpointSetClientFilterConfig /* clientFilterConfig */
            , OverrideYpReqLimitsConfig_ /* overrideYpReqLimitsConfig */
        }
    };
}

void TEndpointSetManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& dependentObjects
    , TVector<ISingleClusterObjectManager::TRequest>& requests
    , TLogFramePtr frame
) const {
    const auto pods = FillPods(dependentObjects.SelectedObjects[0]->Results);
    STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "selected_pods", pods.size());

    for (size_t i = 0; i < EndpointSets_.size(); ++i) {
        TEndpointSetController controller(EndpointSets_[i]);

        STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "selected_endpoints", Endpoints_[i].size());

        TVector<NYP::NClient::TEndpoint> insertEndpoints;
        TVector<TString> removeEndpoints;
        controller.GenerateUpdates(
            Endpoints_[i]
            , pods
            , insertEndpoints
            , removeEndpoints
            , PodsWithPodAgent_
            , frame
        );

        STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "removed_endpoints", removeEndpoints.size());
        STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "inserted_endpoints", insertEndpoints.size());

        for (auto& obj : insertEndpoints) {
            requests.emplace_back(NYP::NClient::TCreateObjectRequest(std::move(obj)));
        }
        for (auto& id : removeEndpoints) {
            requests.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::NApi::NProto::EObjectType::OT_ENDPOINT, std::move(id)));
        }
        if (EndpointSets_[i].Status().controller().has_error()) {
            requests.emplace_back(NYP::NClient::TUpdateRequest(
                NYP::NClient::NApi::NProto::EObjectType::OT_ENDPOINT_SET,
                EndpointSets_[i].Meta().id(),
                {},
                {NYP::NClient::TRemoveRequest("/status/controller/error")}
            ));
        }
    }
}

TVector<NYP::NClient::TPod> TEndpointSetManager::FillPods(
    const TVector<NController::TSelectorResultPtr>& selectResults
) const {
    TVector<NYP::NClient::TPod> pods(selectResults.size());
    for (size_t i = 0; i < pods.size(); ++i) {
        TString readyStatusStr;
        TString targetStateStr;
        selectResults[i]->Fill(
            pods[i].MutableMeta()->mutable_id()
            , pods[i].MutableStatus()->mutable_agent()->mutable_iss_summary()
            , &readyStatusStr
            , pods[i].MutableStatus()->mutable_ip6_address_allocations()
            , pods[i].MutableStatus()->mutable_iss_conf_summaries()
            , &targetStateStr
        );

        {
            static const auto* type = NYT::NYson::ReflectProtobufEnumType(NInfra::NPodAgent::API::EConditionStatus_descriptor());
            if (const auto readyStatusOpt = NYT::NYson::FindProtobufEnumValueByLiteral<NInfra::NPodAgent::API::EConditionStatus>(type, readyStatusStr)) {
                pods[i].MutableStatus()->mutable_agent()->mutable_pod_agent_payload()->mutable_status()->mutable_ready()->set_status(readyStatusOpt.value());
            }
        }

        {
            static const auto* type = NYT::NYson::ReflectProtobufEnumType(NInfra::NPodAgent::API::EPodAgentTargetState_descriptor());
            if (const auto targetStateOpt = NYT::NYson::FindProtobufEnumValueByLiteral<NInfra::NPodAgent::API::EPodAgentTargetState>(type, targetStateStr)) {
                pods[i].MutableSpec()->mutable_pod_agent_payload()->mutable_spec()->set_target_state(targetStateOpt.value());
            }
        }
    }

    Sort(pods, [](const NYP::NClient::TPod& first, const NYP::NClient::TPod& second){return first.Meta().id() < second.Meta().id();});

    return pods;
}

TString TEndpointSetErrorManager::GetObjectId() const {
    return EndpointSetId_;
}

void TEndpointSetErrorManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& /* dependentObjects */
    , TVector<ISingleClusterObjectManager::TRequest>& requests
    , TLogFramePtr /* frame */
) const {
    requests.emplace_back(
        NYP::NClient::TUpdateRequest(
            NYP::NClient::NApi::NProto::OT_ENDPOINT_SET
            , EndpointSetId_
            , {NYP::NClient::TSetRequest("/status/controller/error", Error_)}
            , {} // TRemoveRequest
        )
    );
}

TEndpointSetManagerFactory::TEndpointSetManagerFactory(
    NUpdatableProtoConfig::TAccessor<TEndpointSetManagerFactoryConfig> config
    , NController::TShardPtr shard
)
    : ISingleClusterObjectManagersFactory("enpoint_set_manager_factory", shard, false, Nothing())
    , Config_(std::move(config))
    , ActualConfig_(*Config_)
{
    Config_.SubscribeForUpdate([this](const TEndpointSetManagerFactoryConfig& oldConfig, const TEndpointSetManagerFactoryConfig& newConfig, const NUpdatableProtoConfig::TWatchContext& context = {}) {
        if (context.Id == GetFactoryName() && !google::protobuf::util::MessageDifferencer::Equivalent(oldConfig, newConfig)) {
            ActualConfig_ = newConfig;
        }
    });
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TEndpointSetManagerFactory::GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    NYP::NClient::TSelectObjectsOptions endpointsSelectOptions;
    endpointsSelectOptions.SetLimit(ActualConfig_.GetEndpointsSelectChunkSize());
    TString endpointSetShardFilter = TStringBuilder{} << "farm_hash([/meta/id]) % " << GetShard()->GetNumberOfShards() << " = " << GetShard()->GetShardId();
    TString endpointShardFilter = TStringBuilder{} << "farm_hash([/meta/endpoint_set_id]) % " << GetShard()->GetNumberOfShards() << " = " << GetShard()->GetShardId();
    return {
        {
            NYP::NClient::NApi::NProto::OT_ENDPOINT
            , {
                "/meta/id"
                , "/meta/endpoint_set_id"
                , "/spec"
                , "/status"
            } /* selector */
            , endpointShardFilter /* filter */
            , endpointsSelectOptions /* options */
        }, {
            NYP::NClient::NApi::NProto::OT_ENDPOINT_SET
            , {
                "/meta/id"
                , "/spec"
                , "/labels/supervisor"
                , "/status/controller/error"
            }  /* selector */
            , endpointSetShardFilter /* filter */
            , {} /* options */
        }, {
            NYP::NClient::NApi::NProto::OT_POD
            , {
                "/meta/id"
            } /* selector */
            , "[/spec/pod_agent_payload/meta] != #" /* filter */
            , {} /* options */
        }
    };
}

TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TEndpointSetManagerFactory::TValidationError>> TEndpointSetManagerFactory::GetSingleClusterObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr frame
) const {
    TSimpleTimer timer;
    auto podsWithPodAgent = MakeAtomicShared<THashSet<TString>>();
    for (const auto& podSelectResultPtr : selectorResults[2]->Results) {
        TString podId;
        podSelectResultPtr->Fill(&podId);
        podsWithPodAgent->emplace(podId);
    }

    THashMap<TString, TVector<NYP::NClient::TEndpoint>> endpointSetId2Endpoints;
    for (const auto& endpointSelectResultPtr : selectorResults[0]->Results) {
        NYP::NClient::TEndpoint endpoint;
        endpointSelectResultPtr->Fill(
            endpoint.MutableMeta()->mutable_id()
            , endpoint.MutableMeta()->mutable_endpoint_set_id()
            , endpoint.MutableSpec()
            , endpoint.MutableStatus()
        );

        const TString endpointSetId = endpoint.Meta().endpoint_set_id();
        endpointSetId2Endpoints[endpointSetId].emplace_back(std::move(endpoint));
    }

    TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TEndpointSetManagerFactory::TValidationError>> res;
    THashMap<TString, TVector<NYP::NClient::TEndpointSet>> endpointSets;

    TSet<TString> whitelist;
    {
        bool foundWhitelist = false;
        for (auto& clusterWhitelist : ActualConfig_.GetClientFilterConfig().GetClusterWhitelists()) {
            if (clusterWhitelist.GetCluster() == selectorResults[1]->Cluster) {
                foundWhitelist = true;
                for (const auto& attribute : clusterWhitelist.GetSelectors()) {
                    whitelist.insert(attribute.GetPath());
                }
            }
        }
        if (!foundWhitelist) {
            for (const auto& attribute : ActualConfig_.GetClientFilterConfig().GetCommonWhitelist().GetSelectors()) {
                whitelist.insert(attribute.GetPath());
            }
        }
    }

    for (const auto& endpointSetSelectResultPtr : selectorResults[1]->Results) {
        NYP::NClient::TEndpointSet endpointSet;
        NJson::TJsonValue supervisorValue;
        NYT::TNode controllerError;
        endpointSetSelectResultPtr->Fill(
            endpointSet.MutableMeta()->mutable_id()
            , endpointSet.MutableSpec()
            , &supervisorValue
            , &controllerError
        );

        {
            NYT::TNode curError;
            auto& innerErrors = curError["inner_errors"];
            if (supervisorValue.IsDefined() && (!supervisorValue.IsString() || supervisorValue.GetString() != SERVICE_CONTROLLER_SUPERVISOR)) {
                auto& innerError = innerErrors.Add();
                innerError["message"] = TStringBuilder() << SUPERVISOR_VALUE_ERROR << " " << DOCUMENTATION_LINK;
                innerError["code"] = 0;
            } else {
                if (endpointSet.Spec().pod_filter().empty()) {
                    auto& innerError = innerErrors.Add();
                    innerError["message"] = TStringBuilder() << EMPTY_POD_FILTER_ERROR << " " << DOCUMENTATION_LINK << SERVICE_CONTROLLER_VALIDATION;
                    innerError["code"] = 0;
                } else {
                    try {
                        if (!NYT::NOrm::NLibrary::IntrospectFilterForDefinedAttributeValue(endpointSet.Spec().pod_filter(), "/meta/pod_set_id").has_value()) {
                            STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "filters_not_restricting_pod_set_id", 1);
                            frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TEndpointSetFilterNotRestrictingPodSetId(
                                endpointSet.Meta().id(),
                                endpointSet.Spec().pod_filter(),
                                selectorResults[1]->Cluster
                            ));

                            ValidateFilter(endpointSet.Spec().pod_filter(), whitelist);
                        }
                    } catch (const TWhitelistError& e) {
                        STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "filters_with_forbidden_attributes", 1);
                        frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TEndpointSetFilterWhitelistPathNotFound(
                            endpointSet.Meta().id(),
                            endpointSet.Spec().pod_filter(),
                            selectorResults[1]->Cluster,
                            e.what()
                        ));

                        auto& innerError = innerErrors.Add();
                        innerError["message"] = TStringBuilder() << e.what() << " " << DOCUMENTATION_LINK << SERVICE_CONTROLLER_VALIDATION;
                        innerError["code"] = 0;
                    } catch (...) {
                        STATIC_INFRA_RATE_SENSOR_X(EP_CTL_SENSOR_GROUP, "invalid_pod_filters", 1);
                        frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TEndpointSetInvalidFilter(
                            endpointSet.Meta().id(),
                            endpointSet.Spec().pod_filter(),
                            selectorResults[1]->Cluster
                        ));

                        auto& innerError = innerErrors.Add();
                        innerError["message"] = TStringBuilder() << INVALID_POD_FILTER_ERROR << DOCUMENTATION_LINK;
                        innerError["code"] = 0;
                    }
                }
            }

            if (innerErrors.IsList()) {
                curError["message"] = OMMITTED_SYNC_ERROR;
                curError["code"] = 0; // Errors' codes are unused, setting them just because it is required proto field
                if (curError != controllerError) {
                    res.emplace_back(NController::TSingleClusterObjectManagerPtr(new TEndpointSetErrorManager(
                        endpointSet.Meta().id()
                        , std::move(curError)
                    )));
                }
                continue;
            }
            if (!controllerError.IsUndefined()) {
                YtNodeToProto(controllerError, *endpointSet.MutableStatus()->mutable_controller()->mutable_error());
            }
        }

        const TString podFilter = endpointSet.Spec().pod_filter();
        endpointSets[podFilter].emplace_back(std::move(endpointSet));
    }

    for (auto& it : endpointSets) {
        TVector<TVector<NYP::NClient::TEndpoint>> endpoints;
        for (const auto& endpointSet : it.second) {
            endpoints.emplace_back(std::move(endpointSetId2Endpoints[endpointSet.Meta().id()]));
        }

        res.emplace_back(NController::TSingleClusterObjectManagerPtr(new TEndpointSetManager(
            std::move(it.second)
            , std::move(endpoints)
            , podsWithPodAgent
            , ActualConfig_.GetClientFilterConfig()
            , ActualConfig_.GetOverrideYpReqLimitsConfig()
            , ActualConfig_.GetEndpointSetManager().GetPodsSelectChunkSize()
        )));
    }

    frame->LogEvent(
        ELogPriority::TLOG_INFO
        , NLogEvent::TGetSingleClusterObjectManagersTime(ToString(timer.Get()))
    );
    return res;
}

} // namespace NInfra::NServiceController
