#pragma once

#include <infra/service_controller/libs/config/config.pb.h>

#include <infra/libs/controller/object_manager/object_manager.h>

#include <infra/libs/updatable_proto_config/accessor.h>

namespace NInfra::NServiceController {

class TEndpointSetManager : public NController::ISingleClusterObjectManager {
public:
    TEndpointSetManager(
        TVector<NYP::NClient::TEndpointSet>&& endpointSets
        , TVector<TVector<NYP::NClient::TEndpoint>>&& endpoints
        , const TAtomicSharedPtr<const THashSet<TString>> podsWithPodAgent
        , const NController::TClientFilterConfig& clientFilterConfig
        , const NController::TOverrideYpReqLimitsConfig& overrideYpClientConfig 
        , const ui32 podsSelectChunkSize
    )
        : EndpointSets_(endpointSets)
        , Endpoints_(endpoints)
        , PodsWithPodAgent_(podsWithPodAgent)
        , ClientFilterConfig_(std::move(clientFilterConfig))
        , OverrideYpReqLimitsConfig_(std::move(overrideYpClientConfig))
        , PodsSelectChunkSize_(podsSelectChunkSize)
    {
        Y_ENSURE(!EndpointSets_.empty(), "Endpoint sets list is empty");
        Y_ENSURE(EndpointSets_.size() ==  Endpoints_.size(), "Endpoint sets count must be equal to endpoints lists count");
        for (size_t i = 1; i < EndpointSets_.size(); ++i) {
            Y_ENSURE(EndpointSets_[i].Spec().pod_filter() == EndpointSets_[0].Spec().pod_filter(), "Endpoints' pod filters must be equal to each other");
        }
    }

    virtual TString GetObjectId() const override final;

    virtual TVector<TSelectArgument> GetDependentObjectsSelectArguments() const override final;

    virtual void GenerateYpUpdates(
        const ISingleClusterObjectManager::TDependentObjects& dependentObjects
        , TVector<ISingleClusterObjectManager::TRequest>& requests
        , TLogFramePtr frame
    ) const override final;

private:
    TVector<NYP::NClient::TPod> FillPods(
        const TVector<NController::TSelectorResultPtr>& selectResults
    ) const;

private:
    TVector<NYP::NClient::TEndpointSet> EndpointSets_;
    TVector<TVector<NYP::NClient::TEndpoint>> Endpoints_; // i-th vector - endpoints of EndpointSets_[i]
    const TAtomicSharedPtr<const THashSet<TString>> PodsWithPodAgent_;
    NController::TClientFilterConfig ClientFilterConfig_;
    NController::TOverrideYpReqLimitsConfig OverrideYpReqLimitsConfig_;
    const ui32 PodsSelectChunkSize_;
};

class TEndpointSetErrorManager : public NController::ISingleClusterObjectManager {
public:
    TEndpointSetErrorManager(
        const TString& endpointSetId
        , NYT::TNode&& error
    )
        : EndpointSetId_(endpointSetId)
        , Error_(error)
    {}

    virtual TString GetObjectId() const override final;

    virtual void GenerateYpUpdates(
        const ISingleClusterObjectManager::TDependentObjects& dependentObjects
        , TVector<ISingleClusterObjectManager::TRequest>& requests
        , TLogFramePtr frame
    ) const override final;

private:
    const TString EndpointSetId_;
    const NYT::TNode Error_;
};

class TEndpointSetManagerFactory : public NController::ISingleClusterObjectManagersFactory {
public:
    TEndpointSetManagerFactory(
        NUpdatableProtoConfig::TAccessor<TEndpointSetManagerFactoryConfig> config
        , NController::TShardPtr shard
    );

    virtual TVector<NController::ISingleClusterObjectManager::TSelectArgument> GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */ = {}, NInfra::TLogFramePtr = {}) const override final;

    virtual TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TValidationError>> GetSingleClusterObjectManagers(
        const TVector<NController::TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const override final;

public:
    static constexpr TStringBuf OMMITTED_SYNC_ERROR = "Sync is ommitted.";
    static constexpr TStringBuf EMPTY_POD_FILTER_ERROR = "Pod filter is empty.";
    static constexpr TStringBuf BANNED_POD_FILTER_ERROR = "Pod filter does not restrict [/meta/pod_set_id]. Endpointset created by superuser.";
    static constexpr TStringBuf INVALID_POD_FILTER_ERROR = "Pod filter is invalid.";
    static constexpr TStringBuf SUPERVISOR_VALUE_ERROR = "Supervisor value is unsuitable.";
    static constexpr TStringBuf DOCUMENTATION_LINK = "https://docs.yandex-team.ru/service-controller/";
    static constexpr TStringBuf SERVICE_CONTROLLER_VALIDATION = "#validaciya-endpoint-setov-na-urovne-sc";

private:
    NUpdatableProtoConfig::TAccessor<TEndpointSetManagerFactoryConfig> Config_;
    TEndpointSetManagerFactoryConfig ActualConfig_;
};

} // namespace NInfra::NServiceController
