#include "controller.h"

#include <infra/libs/logger/logger.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NServiceController {

namespace {

template<typename TObject>
TString ToYson(const TObject& obj) {
    return NYT::NYson::ConvertToYsonString(obj).ToString();
}

TLogger LOGGER({});
const TEndpointSetManagerFactoryConfig FACTORY_CONFIG;

NController::TSelectObjectsResultPtr CreateSelectObjectsResultPtr(const TVector<NController::TSelectorResultPtr>& selectorResults) {
    return MakeAtomicShared<NController::TSelectObjectsResult>(selectorResults);
}
} // namespace

Y_UNIT_TEST_SUITE(EndpointSetManagerFactoryTest) {

Y_UNIT_TEST(FactorySelectArgument) {
    auto startCfg = FACTORY_CONFIG;
    startCfg.SetEndpointsSelectChunkSize(123);
    NUpdatableProtoConfig::TStaticConfigHolder<TEndpointSetManagerFactoryConfig> cfgHolder(startCfg);
    auto cfg = cfgHolder.Accessor();

    NController::TSharding shardFactory; 
    TEndpointSetManagerFactory factory(cfg, shardFactory.GetShard(0));
    auto selectArgs = factory.GetSelectArgumentsSafe({}, {});
    UNIT_ASSERT_EQUAL_C(3, selectArgs.size(), selectArgs.size());

    {
        auto selectArg = selectArgs[0];
        UNIT_ASSERT_EQUAL(selectArg.ObjectType, NYP::NClient::NApi::NProto::OT_ENDPOINT);
        UNIT_ASSERT_EQUAL_C(4, selectArg.Selector.size(), selectArg.Selector.size());
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[0], "/meta/id", selectArg.Selector[0]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[1], "/meta/endpoint_set_id", selectArg.Selector[1]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[2], "/spec", selectArg.Selector[2]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[3], "/status", selectArg.Selector[3]);
        UNIT_ASSERT_EQUAL_C(selectArg.Options.Limit(), CONFIG_SNAPSHOT_VALUE(cfg, GetEndpointsSelectChunkSize()), selectArg.Options.Limit());
    }

    {
        auto selectArg = selectArgs[1];
        UNIT_ASSERT_EQUAL(selectArg.ObjectType, NYP::NClient::NApi::NProto::OT_ENDPOINT_SET);
        UNIT_ASSERT_EQUAL_C(4, selectArg.Selector.size(), selectArg.Selector.size());
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[0], "/meta/id", selectArg.Selector[0]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[1], "/spec", selectArg.Selector[1]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[2], "/labels/supervisor", selectArg.Selector[2]);
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[3], "/status/controller/error", selectArg.Selector[3]);
    }

    {
        auto selectArg = selectArgs[2];
        UNIT_ASSERT_EQUAL(selectArg.ObjectType, NYP::NClient::NApi::NProto::OT_POD);
        UNIT_ASSERT_EQUAL_C(1, selectArg.Selector.size(), selectArg.Selector.size());
        UNIT_ASSERT_EQUAL_C(selectArg.Selector[0], "/meta/id", selectArg.Selector[0]);
        UNIT_ASSERT_EQUAL_C(selectArg.Filter, "[/spec/pod_agent_payload/meta] != #", selectArg.Filter);
    }
}

Y_UNIT_TEST(EndpointSetErrorManagerUpdateTest) {
    NYP::NClient::NApi::NProto::TAttributeList endpointSetSelectResult;
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson("some_id")); // /meta/id
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /spec
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson("abcd")); // /labels/supervisor
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /status/controller/error

    NYP::NClient::NApi::NProto::TAttributeList endpointSetSelectResultWithBadLabel;
    endpointSetSelectResultWithBadLabel.add_value_payloads()->set_yson(ToYson("other_id")); // /meta/id
    endpointSetSelectResultWithBadLabel.add_value_payloads()->set_yson("{\"pod_filter\"=\"[/labels/a]=1\"}"); // /spec
    endpointSetSelectResultWithBadLabel.add_value_payloads()->set_yson(ToYson("service-controller")); // /labels/supervisor
    endpointSetSelectResultWithBadLabel.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /status/controller/error

    NYP::NClient::NApi::NProto::TAttributeList endpointSetSelectResultWithEmptyFilter;
    endpointSetSelectResultWithEmptyFilter.add_value_payloads()->set_yson(ToYson("and_another_id")); // /meta/id
    endpointSetSelectResultWithEmptyFilter.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /spec
    endpointSetSelectResultWithEmptyFilter.add_value_payloads()->set_yson(ToYson("service-controller")); // /labels/supervisor
    endpointSetSelectResultWithEmptyFilter.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /status/controller/error

    NUpdatableProtoConfig::TStaticConfigHolder<TEndpointSetManagerFactoryConfig> cfgHolder(FACTORY_CONFIG);
    auto cfg = cfgHolder.Accessor();

    NController::TSharding shardFactory;
    TEndpointSetManagerFactory factory(cfg, shardFactory.GetShard(0));

    auto emptySelectObjectsResultPtr = CreateSelectObjectsResultPtr({});
    auto res = factory.GetSingleClusterObjectManagers({
        emptySelectObjectsResultPtr,
        CreateSelectObjectsResultPtr({
            MakeAtomicShared<NYP::NClient::TSelectorResult>(endpointSetSelectResult),
            MakeAtomicShared<NYP::NClient::TSelectorResult>(endpointSetSelectResultWithBadLabel),
            MakeAtomicShared<NYP::NClient::TSelectorResult>(endpointSetSelectResultWithEmptyFilter)}),
        emptySelectObjectsResultPtr}, LOGGER.SpawnFrame());

    {
        UNIT_ASSERT_EQUAL_C(3, res.size(), res.size());
        UNIT_ASSERT(res[0].Success());
        UNIT_ASSERT(res[1].Success());
        UNIT_ASSERT(res[2].Success());
    }

    auto manager = res[0].Success();
    TVector<NController::ISingleClusterObjectManager::TRequest> requests;
    manager->GenerateYpUpdates({}, requests, LOGGER.SpawnFrame());

    auto managerWithBadLabel = res[1].Success();
    TVector<NController::ISingleClusterObjectManager::TRequest> requestsWithBadLabel;
    managerWithBadLabel->GenerateYpUpdates({}, requestsWithBadLabel, LOGGER.SpawnFrame());

    auto managerWithEmptyFilter = res[2].Success();
    TVector<NController::ISingleClusterObjectManager::TRequest> requestsWithEmptyFilter;
    managerWithEmptyFilter->GenerateYpUpdates({}, requestsWithEmptyFilter, LOGGER.SpawnFrame());

    UNIT_ASSERT_EQUAL_C(1, requests.size(), requests.size());
    UNIT_ASSERT(std::holds_alternative<NYP::NClient::TUpdateRequest>(requests[0]));
    auto& req = std::get<NYP::NClient::TUpdateRequest>(requests[0]);
    UNIT_ASSERT_EQUAL(req.GetObjectType(), NYP::NClient::NApi::NProto::OT_ENDPOINT_SET);
    UNIT_ASSERT_EQUAL_C("some_id", req.GetObjectId(), req.GetObjectId());

    UNIT_ASSERT_EQUAL_C(1, requestsWithBadLabel.size(), requestsWithBadLabel.size());
    UNIT_ASSERT(std::holds_alternative<NYP::NClient::TUpdateRequest>(requestsWithBadLabel[0]));
    auto& reqWithBadLabel = std::get<NYP::NClient::TUpdateRequest>(requestsWithBadLabel[0]);
    UNIT_ASSERT_EQUAL(reqWithBadLabel.GetObjectType(), NYP::NClient::NApi::NProto::OT_ENDPOINT_SET);
    UNIT_ASSERT_EQUAL_C("other_id", reqWithBadLabel.GetObjectId(), reqWithBadLabel.GetObjectId());

    UNIT_ASSERT_EQUAL_C(1, requestsWithEmptyFilter.size(), requestsWithEmptyFilter.size());
    UNIT_ASSERT(std::holds_alternative<NYP::NClient::TUpdateRequest>(requestsWithEmptyFilter[0]));
    auto& reqWithEmptyFilter = std::get<NYP::NClient::TUpdateRequest>(requestsWithEmptyFilter[0]);
    UNIT_ASSERT_EQUAL(reqWithEmptyFilter.GetObjectType(), NYP::NClient::NApi::NProto::OT_ENDPOINT_SET);
    UNIT_ASSERT_EQUAL_C("and_another_id", reqWithEmptyFilter.GetObjectId(), reqWithEmptyFilter.GetObjectId());

    UNIT_ASSERT_EQUAL_C(1, req.GetSetVec().size(), req.GetSetVec().size());
    UNIT_ASSERT_EQUAL_C("/status/controller/error", req.GetSetVec()[0].GetPath(), req.GetSetVec()[0].GetPath());
    auto updateValue = NYT::NodeFromYsonString(req.GetSetVec()[0].GetValue());
    UNIT_ASSERT_EQUAL_C(TEndpointSetManagerFactory::OMMITTED_SYNC_ERROR, updateValue["message"].AsString(), updateValue["message"].AsString());

    UNIT_ASSERT_EQUAL_C(1, reqWithBadLabel.GetSetVec().size(), reqWithBadLabel.GetSetVec().size());
    UNIT_ASSERT_EQUAL_C("/status/controller/error", reqWithBadLabel.GetSetVec()[0].GetPath(), reqWithBadLabel.GetSetVec()[0].GetPath());
    auto updateValueWithBadLabel = NYT::NodeFromYsonString(reqWithBadLabel.GetSetVec()[0].GetValue());
    UNIT_ASSERT_EQUAL_C(TEndpointSetManagerFactory::OMMITTED_SYNC_ERROR, updateValueWithBadLabel["message"].AsString(), updateValueWithBadLabel["message"].AsString());

    UNIT_ASSERT_EQUAL_C(1, reqWithEmptyFilter.GetSetVec().size(), reqWithEmptyFilter.GetSetVec().size());
    UNIT_ASSERT_EQUAL_C("/status/controller/error", reqWithEmptyFilter.GetSetVec()[0].GetPath(), reqWithEmptyFilter.GetSetVec()[0].GetPath());
    auto updateValueWithEmptyFilter = NYT::NodeFromYsonString(reqWithEmptyFilter.GetSetVec()[0].GetValue());
    UNIT_ASSERT_EQUAL_C(TEndpointSetManagerFactory::OMMITTED_SYNC_ERROR, updateValueWithEmptyFilter["message"].AsString(), updateValueWithEmptyFilter["message"].AsString());

    UNIT_ASSERT_EQUAL_C(1, updateValue["inner_errors"].AsList().size(), updateValue["inner_errors"].AsList().size());
    TSet<TString> innerMessages;
    for (auto& innerError : updateValue["inner_errors"].AsList()) {
        UNIT_ASSERT(innerMessages.emplace(innerError["message"].AsString()).second);
    }
    UNIT_ASSERT(innerMessages.contains(TStringBuilder() << TEndpointSetManagerFactory::SUPERVISOR_VALUE_ERROR << " " << TEndpointSetManagerFactory::DOCUMENTATION_LINK));

    UNIT_ASSERT_EQUAL_C(1, updateValueWithBadLabel["inner_errors"].AsList().size(), updateValueWithBadLabel["inner_errors"].AsList().size());
    TSet<TString> innerMessagesWithBadLabel;
    for (auto& innerError : updateValueWithBadLabel["inner_errors"].AsList()) {
        UNIT_ASSERT(innerMessagesWithBadLabel.emplace(innerError["message"].AsString()).second);
    }
    UNIT_ASSERT(innerMessagesWithBadLabel.contains(TStringBuilder()
        << "Selector [/labels/a] in filter is not in whitelist. "
        << TEndpointSetManagerFactory::DOCUMENTATION_LINK << TEndpointSetManagerFactory::SERVICE_CONTROLLER_VALIDATION));

    UNIT_ASSERT_EQUAL_C(1, updateValueWithEmptyFilter["inner_errors"].AsList().size(), updateValueWithEmptyFilter["inner_errors"].AsList().size());
    TSet<TString> innerMessagesWithEmptyFilter;
    for (auto& innerError : updateValueWithEmptyFilter["inner_errors"].AsList()) {
        UNIT_ASSERT(innerMessagesWithEmptyFilter.emplace(innerError["message"].AsString()).second);
    }
    UNIT_ASSERT(innerMessagesWithEmptyFilter.contains(TStringBuilder()
        << TEndpointSetManagerFactory::EMPTY_POD_FILTER_ERROR << " " << TEndpointSetManagerFactory::DOCUMENTATION_LINK << TEndpointSetManagerFactory::SERVICE_CONTROLLER_VALIDATION));
}

Y_UNIT_TEST(EndpointSetErrorManagerUpdateOnlyOnChangeTest) {
    NYP::NClient::NApi::NProto::TAttributeList endpointSetSelectResult;
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson("some_id")); // /meta/id
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /spec
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson("abcd")); // /labels/supervisor
    endpointSetSelectResult.add_value_payloads()->set_yson(ToYson(THashMap<TString, TString>())); // /status/controller/error

    NUpdatableProtoConfig::TStaticConfigHolder<TEndpointSetManagerFactoryConfig> cfgHolder(FACTORY_CONFIG);
    auto cfg = cfgHolder.Accessor();

    NController::TSharding shardFactory;
    TEndpointSetManagerFactory factory(cfg, shardFactory.GetShard(0));

    auto emptySelectObjectsResultPtr = CreateSelectObjectsResultPtr({});
    auto res = factory.GetSingleClusterObjectManagers({emptySelectObjectsResultPtr, CreateSelectObjectsResultPtr({MakeAtomicShared<NYP::NClient::TSelectorResult>(endpointSetSelectResult)}), emptySelectObjectsResultPtr}, LOGGER.SpawnFrame());
    TVector<NController::ISingleClusterObjectManager::TRequest> requests;
    res[0].Success()->GenerateYpUpdates({}, requests, LOGGER.SpawnFrame());

    auto& req = std::get<NYP::NClient::TUpdateRequest>(requests[0]);
    auto updateValue = NYT::NodeFromYsonString(req.GetSetVec()[0].GetValue());

    endpointSetSelectResult.mutable_value_payloads(3)->set_yson(NYT::NodeToYsonString(updateValue));
    res = factory.GetSingleClusterObjectManagers({emptySelectObjectsResultPtr, CreateSelectObjectsResultPtr({MakeAtomicShared<NYP::NClient::TSelectorResult>(endpointSetSelectResult)}), emptySelectObjectsResultPtr}, LOGGER.SpawnFrame());
    UNIT_ASSERT_C(res.empty(), res.size());
}

}

} // namespace NInfra::NServiceController
