#include "endpoint_set_controller.h"

#include <infra/libs/sensors/macros.h>
#include <yp/cpp/yp/yson_interop.h>

#include <yt/yt/core/yson/public.h>

#include <util/string/builder.h>
#include <util/system/env.h>

namespace NInfra::NServiceController {

namespace {

constexpr std::array<NPodAgent::API::EPodAgentTargetState, 2> UNSUITABLE_POD_AGENT_TARGET_STATES = {
    NPodAgent::API::EPodAgentTargetState_REMOVED,
    NPodAgent::API::EPodAgentTargetState_SUSPENDED,
};

TMaybe<TEndpoint> BuildEndpoint(const NYP::NClient::TEndpointSet& endpointSet, const NYP::NClient::TPod& pod) {
    TMaybe<TEndpoint> endpoint;
    for (const auto& alloc : pod.Status().ip6_address_allocations()) {
        if (alloc.vlan_id() == "backbone") {
            if (alloc.has_persistent_fqdn() || alloc.has_transient_fqdn()) {
                return TEndpoint(endpointSet, pod, alloc);
            }
            if (!endpoint.Defined()) {
                endpoint = TEndpoint(endpointSet, pod, alloc);
            }
        }
    }
    return endpoint;
}

bool IsAlive(const NYP::NClient::TPod& pod, const bool podWithPodAgent) {
    if (podWithPodAgent) {
        bool podStatusAlive = false;
        if (pod.Status().agent().pod_agent_payload().has_status() &&
            pod.Status().agent().pod_agent_payload().status().has_ready()) {
            const auto& status = pod.Status().agent().pod_agent_payload().status().ready().status();
            podStatusAlive = (status == NPodAgent::API::EConditionStatus_TRUE);
        }
        return podStatusAlive;
    } else {
        for (const auto& [_, v] : pod.Status().agent().iss_summary().state_summaries()) {
            if (v.current_state() == "ACTIVE") {
                return true;
            }
        }
    }

    return false;
}

bool DeployTargetStateIsSuitableForEndpoint(const NYP::NClient::TPod& pod) {
    // function should be used to determine when we must remove endpoint
    // now we assume that pods with all states exclude "removed" or "suspended" are active (not only with "active" state)
    const NPodAgent::API::EPodAgentTargetState& targetState = pod.Spec().pod_agent_payload().spec().target_state();
    return Find(UNSUITABLE_POD_AGENT_TARGET_STATES, targetState) == UNSUITABLE_POD_AGENT_TARGET_STATES.end();
}

bool YpliteTargetStateIsActive(const NYP::NClient::TPod& pod) {
    for (const auto& [_, v] : pod.Status().iss_conf_summaries()) {
        if (v.target_state() == "ACTIVE") {
            return true;
        }
    }

    return false;
}

} // namespace

void TEndpointSetController::GenerateUpdates(
    const TVector<NYP::NClient::TEndpoint>& ypEndpoints
    , const TVector<NYP::NClient::TPod>& pods
    , TVector<NYP::NClient::TEndpoint>& insert
    , TVector<TString>& remove
    , const TAtomicSharedPtr<const THashSet<TString>> podsWithPodAgent
    , TLogFramePtr frame
) {
    THashMap<TEndpointKey, TEndpointData, TEndpointKey::THash, TEndpointKey::TEqualTo> endpoints;

    TVector<TVector<TEndpoint>> buildedEndpoints(2); // "unready"/"ready" classfication
    NLogEvent::TEndpointSetInfo endpointSetInfo;
    endpointSetInfo.SetId(EndpointSet_.Meta().id());
    // sync pods
    for (const auto& pod : pods) {
        TMaybe<TEndpoint> endpoint = BuildEndpoint(EndpointSet_, pod);
        auto podInfo = endpointSetInfo.AddPodsInfo();
        podInfo->SetId(pod.Meta().id());
        podInfo->SetEndpointBuildIsSuccess(endpoint.Defined());

        if (!endpoint.Defined()) {
            continue;
        }

        const bool podWithPodAgent = podsWithPodAgent->contains(pod.Meta().id());

        podInfo->SetIsAlive(IsAlive(pod, podWithPodAgent));
        podInfo->SetPodWithPodAgent(podWithPodAgent);

        bool ready;
        if (podWithPodAgent) {
            // for deploy need AND in condition, see YP-2705
            podInfo->SetTargetStateIsActive(DeployTargetStateIsSuitableForEndpoint(pod));
            ready = podInfo->GetIsAlive() && DeployTargetStateIsSuitableForEndpoint(pod);
        } else {
            // for nanny need OR in condition, see YP-1501
            podInfo->SetTargetStateIsActive(YpliteTargetStateIsActive(pod));
            ready = podInfo->GetIsAlive() || podInfo->GetTargetStateIsActive();
        }
        endpoint->SetReady(ready);
        buildedEndpoints[ready].emplace_back(std::move(*endpoint));
    }

    frame->LogEvent(ELogPriority::TLOG_DEBUG, endpointSetInfo);

    const size_t requiredEndpointsCount = ceil(EndpointSet_.Spec().liveness_limit_ratio() * pods.size());
    for (const bool ready : {true, false}) {
        for (const auto& endpoint : buildedEndpoints[ready]) {
            if (ready || endpoints.size() < requiredEndpointsCount) {
                endpoints[endpoint.Key()] = endpoint.Data();
            }
        }
    }

    //sync endpoints
    for (const auto& ypEndpoint : ypEndpoints) {
        TEndpoint endpoint;
        endpoint.Load(ypEndpoint);
        endpoint.Data().DropPodRelation();

        if (auto* mappedEndpoint = endpoints.FindPtr(endpoint.Key())) {
            if (mappedEndpoint->IsSyncedToYP()) {
                remove.push_back(endpoint.Data().EndpointId());
            } else {
                mappedEndpoint->Sync(endpoint.Data());
            }
        } else {
            remove.push_back(endpoint.Data().EndpointId());
        }
    }

    for (const auto& p : endpoints) {
        TEndpoint endpoint(p.first, p.second);

        if (!endpoint.Data().IsSyncedToYP()) {
            NYP::NClient::TEndpoint ypEndpoint;
            endpoint.Save(ypEndpoint);
            insert.push_back(ypEndpoint);
        }
    }
}

} // namespace NInfra::NServiceController
