#include "endpoint_set_controller.h"

#include <infra/libs/logger/logger.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/yson/node/node.h>

static auto LOG_FRAME = NInfra::TLogger({}).SpawnFrame();

static const auto PODS_WITH_POD_AGENT = MakeAtomicShared<const THashSet<TString>>();

using namespace NInfra::NServiceController;

TEndpointSetController CreateController() {
    NYP::NClient::TEndpointSet set;
    set.MutableMeta()->set_id("endpoint_set_1");
    set.MutableSpec()->set_pod_filter("[/meta/pod_set_id] = 'pod_set_id'");
    set.MutableSpec()->set_port(3388);
    set.MutableSpec()->set_protocol("TCP");

    return TEndpointSetController(set);
}

NYP::NClient::TPod CreatePod(
    const TString& id
    , const TMaybe<TString>& ip4Address
    , const TString& ip6Address
    , bool isActive
) {
    NYP::NClient::TPod pod;
    pod.MutableMeta()->set_id(id);
    auto resultIssSummary = pod.MutableStatus()->mutable_agent()->mutable_iss_summary();
    (*resultIssSummary->mutable_state_summaries())["test_configuration"].set_current_state(isActive ? "ACTIVE" : "PREPARED");
    auto& ip6AddressAllocation = *pod.MutableStatus()->add_ip6_address_allocations();
    ip6AddressAllocation.set_address(ip6Address);
    ip6AddressAllocation.set_vlan_id("backbone");
    if (ip4Address.Defined()) {
        ip6AddressAllocation.mutable_internet_address()->set_ip4_address(*ip4Address);
    }

    return pod;
}

void SyncToYP(TVector<NYP::NClient::TEndpoint>& ypEndpoints, const TVector<NYP::NClient::TEndpoint>& endpoints,
              const TVector<TString>& removes, size_t offset) {
    THashMap<TString, NYP::NClient::TEndpoint> stor;
    for (const auto& e : ypEndpoints) {
        stor[e.Meta().id()] = e;
    }

    for (size_t i = 0; i < endpoints.size(); ++i) {
        NYP::NClient::TEndpoint e(endpoints[i]);
        e.MutableMeta()->set_id(ToString(offset * 100500 + i));

        Y_VERIFY(!stor.contains(e.Meta().id()));
        stor[e.Meta().id()] = e;
    }

    for (const TString& id : removes) {
        Y_VERIFY(stor.contains(id));
        stor.erase(id);
    }

    ypEndpoints.clear();
    for (const auto& pt : stor) {
        ypEndpoints.push_back(pt.second);
    }
}

Y_UNIT_TEST_SUITE(EndpointSetControllerTest) {
    Y_UNIT_TEST(SimpleTest) {
        TVector<NYP::NClient::TEndpoint> ypEndpoints;

        auto ctl = CreateController();

        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            const TString ip4Address = "1:1:1:1";
            const TString ip6Address = "2:2:2:2";

            ctl.GenerateUpdates(
                ypEndpoints
                , {CreatePod("pod_id_1", ip4Address, ip6Address, true)}
                , endpoints
                , removes
                , PODS_WITH_POD_AGENT
                , LOG_FRAME
            );

            UNIT_ASSERT_VALUES_EQUAL(1, endpoints.size());
            UNIT_ASSERT_STRINGS_EQUAL(endpoints[0].Spec().Getip4_address(), ip4Address);
            UNIT_ASSERT_STRINGS_EQUAL(endpoints[0].Spec().Getip6_address(), ip6Address);

            UNIT_ASSERT_VALUES_EQUAL(0, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }

        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            ctl.GenerateUpdates(ypEndpoints, {}, endpoints, removes, PODS_WITH_POD_AGENT, LOG_FRAME);

            UNIT_ASSERT_VALUES_EQUAL(0, endpoints.size());
            UNIT_ASSERT_VALUES_EQUAL(1, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }

        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            ctl.GenerateUpdates(
                ypEndpoints
                , {CreatePod("pod_id_2", "1:1:1:1", "2:2:2:2", false), CreatePod("pod_id_3", "3:3:3:3", "4:4:4:4", true)}
                , endpoints
                , removes
                , PODS_WITH_POD_AGENT
                , LOG_FRAME
            );

            UNIT_ASSERT_VALUES_EQUAL(1, endpoints.size());
            UNIT_ASSERT_VALUES_EQUAL(0, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }

        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            ctl.GenerateUpdates(
                ypEndpoints
                , {CreatePod("pod_id_3", "1:1:1:1", "2:2:2:2", false)}
                , endpoints
                , removes
                , PODS_WITH_POD_AGENT
                , LOG_FRAME
            );

            UNIT_ASSERT_VALUES_EQUAL(0, endpoints.size());
            UNIT_ASSERT_VALUES_EQUAL(1, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }

        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            ctl.GenerateUpdates(
                ypEndpoints
                , {CreatePod("pod_id_3", "1:1:1:1", "2:2:2:2", false)}
                , endpoints
                , removes
                , PODS_WITH_POD_AGENT
                , LOG_FRAME
            );

            UNIT_ASSERT_VALUES_EQUAL(0, endpoints.size());
            UNIT_ASSERT_VALUES_EQUAL(0, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }

        // Test pod without ip4 address
        {
            TVector<NYP::NClient::TEndpoint> endpoints;
            TVector<TString> removes;

            const TString ip6Address = "2:2:2:2";

            ctl.GenerateUpdates(
                ypEndpoints
                , {CreatePod("pod_id_1", Nothing(), ip6Address, true)}
                , endpoints
                , removes
                , PODS_WITH_POD_AGENT
                , LOG_FRAME
            );

            UNIT_ASSERT_VALUES_EQUAL(1, endpoints.size());
            UNIT_ASSERT(!endpoints[0].Spec().has_ip4_address());
            UNIT_ASSERT_STRINGS_EQUAL(endpoints[0].Spec().Getip6_address(), ip6Address);

            UNIT_ASSERT_VALUES_EQUAL(0, removes.size());

            SyncToYP(ypEndpoints, endpoints, removes, __LINE__);
        }
    }
}
