#include "main.h"

#include <cstdlib>

#include <infra/libs/logger/event_iterator.h>
#include <infra/libs/logger/protos/events.ev.pb.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/scheme/scheme.h>

namespace NInfra::NServiceController {

namespace {

template <typename TEvent>
bool Is(const NProtoBuf::Message* proto) {
    return TEvent().GetTypeName() == proto->GetTypeName();
}

void JumpToLastStart(TCachedEventIterator& iter) {
    while (iter.SafeNext());
    while (const TEvent* event = iter.SafePrev()) {
        if (Is<NLogEvent::TStartService>(event->GetProto())) {
            return;
        }
    }
}

} // namespace

int RunPrintDiff(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<eventlog>", "Event log file");

    NLastGetopt::TOptsParseResult parsedOpts(&opts, argc, argv);
    TIteratorOptions options({.FileName = parsedOpts.GetFreeArgs()[0]});
    TCachedEventIterator iter(options);

    JumpToLastStart(iter);

    NSc::TValue result;
    NSc::TValue gotResult;
    NSc::TValue factoriesStart;
    size_t numberOfUnfinishedFactories = 0;

    bool wasStart = false;
    while (const TEvent* event = iter.SafeNext()) {
        const NProtoBuf::Message* proto = event->GetProto();

        if (Is<NLogEvent::TStartSyncCycle>(proto)) {
            auto ptr = dynamic_cast<const NLogEvent::TStartSyncCycle*>(proto);
            const TString factoryName = ptr->GetFactoryName();
            if (!factoriesStart[factoryName].GetBool()) {
                if (!wasStart) {
                    result.Clear();
                    result["insert"]["ids"].SetArray();
                    result["insert"]["endpoints"].SetArray();
                    result["remove"]["ids"].SetArray();
                }

                gotResult[factoryName] = false;
                factoriesStart[factoryName] = true;
                wasStart = true;
                ++numberOfUnfinishedFactories;
            }
        }

        if (!wasStart) {
            continue;
        }

        if (Is<NLogEvent::TCreateYPObject>(proto)) {
            auto ptr = dynamic_cast<const NLogEvent::TCreateYPObject*>(proto);
            const TString factoryName = ptr->GetFactoryName();
            if (ptr->GetType() == NLogEvent::EYPObjectType::OT_ENDPOINT && !gotResult[factoryName].GetBool()) {
                result["insert"]["ids"].Push(ptr->GetId());
                result["insert"]["endpoints"].Push(ptr->GetAttributes());
            }
        } else if (Is<NLogEvent::TRemoveYPObject>(proto)) {
            auto ptr = dynamic_cast<const NLogEvent::TRemoveYPObject*>(proto);
            const TString factoryName = ptr->GetFactoryName();
            if (ptr->GetType() == NLogEvent::EYPObjectType::OT_ENDPOINT && !gotResult[factoryName].GetBool()) {
                result["remove"]["ids"].Push(ptr->GetId());
            }
        } else if (Is<NLogEvent::TSyncCycleSuccess>(proto)) {
            const TString factoryName = dynamic_cast<const NLogEvent::TSyncCycleSuccess*>(proto)->GetFactoryName();
            if (!gotResult[factoryName].GetBool()) {
                --numberOfUnfinishedFactories;
            }
            gotResult[factoryName] = true;
        } else if (Is<NLogEvent::TSyncLoopError>(proto)) {
            Cerr << "Sync loop error." << Endl;
            return EXIT_FAILURE;
        }
    }

    if (!wasStart) {
        Cerr << "TStartSyncCycle event was not found" << Endl;
        return EXIT_FAILURE;
    }

    if (numberOfUnfinishedFactories) {
        Cerr << "Could not match all TStartSyncCycle and TSyncCycleSuccess logs" << Endl;
        return EXIT_FAILURE;
    }

    Cout << result.ToJson(NSc::TJsonOpts::JO_SORT_KEYS | NSc::TJsonOpts::JO_PARSER_STRICT_UTF8) << Endl;
    return EXIT_SUCCESS;
}

} // namespace NInfra::NServiceController
