import argparse
import json
import juggler_sdk
import logging
import sys

from infra.yp_service_discovery.monitoring.solomon.src.util.solomon_api import SolomonApi

SOLOMON_PROJECT_ID = 'service_controller'

JUGGLER_NAMESPACE = 'yp.service_controller'
JUGGLER_HOST = 'SERVICE_CONTROLLER'

WARN = 'warn'
CRIT = 'crit'

MARK = 'service-controller'

LOGINS = {
    WARN: [
        '@svc_service_discovery:yp-service-discovery-duty',
        'dns-monitoring',
    ],
    CRIT: [
        '@svc_yp_dns:yp-dns-duty',
        'dns-monitoring',
    ],
}

PHONE_ESCALATION_LOGINS = {
    WARN: [
    ],
    CRIT: [
        '@svc_yp_dns:yp-dns-duty',
        'ismagilas',
        'azuremint',
        'elshiko',
        'avitella',
    ],
}

NOTIFICATION_ALWAYS = {
    "day_start": 1,
    "day_end": 7,
    "time_start": 0,
    "time_end": 23,
}

NOTIFICATION_WORK_TIME = {
    "day_start": 1,
    "day_end": 5,
    "time_start": 10,
    "time_end": 21,
}


def milliseconds(seconds=0, minutes=0):
    return seconds * 1000 + minutes * 60 * 1000


def get_solomon_expression_alert(name: str, title_name: str,
                                 program: str, check_expression: str,
                                 annotations: dict = {},
                                 group_by_labels: list = [],
                                 period_ms: int = milliseconds(minutes=3),
                                 delay_s: int = 30,
                                 resolved_empty_policy: str = "RESOLVED_EMPTY_DEFAULT",
                                 no_points_policy: str = "NO_POINTS_DEFAULT"):
    return {
        'id': name,
        'projectId': SOLOMON_PROJECT_ID,
        'name': title_name,
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'expression': {
                'program': program,
                'checkExpression': check_expression,
            }
        },
        'groupByLabels': group_by_labels,
        'periodMillis': period_ms,
        'delaySeconds': delay_s,
        'annotations': annotations,
        'resolvedEmptyPolicy': resolved_empty_policy,
        'noPointsPolicy': no_points_policy,
    }


def milliseconds(seconds=0, minutes=0):
    return seconds * 1000 + minutes * 60 * 1000


def notification(status, logins, notify_methods, time_limits=NOTIFICATION_ALWAYS):
    template_kwargs = {
        "status": status,
        "login": logins,
        "method": notify_methods,
        "day_start": time_limits["day_start"],
        "day_end": time_limits["day_end"],
        "time_start": time_limits["time_start"],
        "time_end": time_limits["time_end"],
    }

    if (isinstance(status, str) and status == "CRIT") or (isinstance(status, dict) and status["to"] == "CRIT"):
        template_kwargs["repeat"] = 1800  # 30 min

    return juggler_sdk.NotificationOptions(
        template_name="on_status_change",
        template_kwargs=template_kwargs,
    )


def get_notifications_config(level=CRIT, time_limits=NOTIFICATION_ALWAYS):
    logins = LOGINS[level]
    escalation_logins = PHONE_ESCALATION_LOGINS[level]

    result = []
    if logins:
        result.extend([
            notification(
                status={'from': 'OK', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ],
                time_limits=time_limits
            ),
            notification(
                status={'from': 'WARN', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ],
                time_limits=time_limits
            ),
            notification(
                status={'from': 'CRIT', 'to': 'OK'},
                logins=logins,
                notify_methods=[
                    'telegram',
                ],
                time_limits=time_limits
            ),
        ])

    if escalation_logins:
        result.extend([
            juggler_sdk.NotificationOptions(
                template_name='phone_escalation',
                template_kwargs={
                    'delay': 60,
                    'logins': escalation_logins,
                    'day_start': time_limits['day_start'],
                    'day_end': time_limits['day_end'],
                    'time_start': time_limits['time_start'],
                    'time_end': time_limits['time_end'],
                }
            ),
        ])

    return result


def get_juggler_service(service_id):
    return '.'.join(['yp', 'service_controller', service_id])


def get_yp_downtime_tags(yp_cluster):
    return [
        "yp-downtime-9b8ee2a2b559b3afff751623e5e79546",
        "yp-{}-downtime-9b8ee2a2b559b3afff751623e5e79546".format(yp_cluster),
    ]


def get_solomon_lag_checks():
    return ['solomon:{}'.format(get_juggler_service('solomon_lag'))]


def get_yp_unreach_checks(yp_cluster):
    return ['yp-{}.yp.yandex.net:yp.infra.auto_downtime_'.format(yp_cluster)]


def get_yp_standard_unreach_checks(yp_cluster):
    return get_yp_unreach_checks(yp_cluster) + get_solomon_lag_checks()


def get_solomon_alert_url(alert_id):
    return {
        'url': 'https://solomon.yandex-team.ru/admin/projects/{}/alerts/{}'.format(SOLOMON_PROJECT_ID, alert_id),
        'title': 'Alert on solomon',
    }


def get_nanny_service_url(yp_cluster):
    return {
        'url': 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}_service_controller'.format(yp_cluster),
        'title': 'Nanny service',
        'type': 'nanny',
    }


def get_source_url():
    return {
        'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/infra/service_controller/monitoring/alerts',
        'title': 'Alerts generation source code',
    }


def get_standard_urls(alert_id, solomon_cluster):
    return [
        get_solomon_alert_url(alert_id),
        get_nanny_service_url(solomon_cluster['name']),
        get_source_url(),
    ]


def get_juggler_check_with_yp_downtime(host, service, yp_cluster,
                                       notification_time_limits=NOTIFICATION_ALWAYS,
                                       notification_level=CRIT,
                                       ttl=900,
                                       flaps_config=juggler_sdk.FlapOptions(stable=180, critical=300, boost=0)):
    return juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        tags=get_yp_downtime_tags(yp_cluster),
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': get_yp_standard_unreach_checks(yp_cluster),
        },
        ttl=ttl,
        notifications=get_notifications_config(notification_level, notification_time_limits),
        flaps_config=flaps_config,
    )


def get_juggler_check_downtime(host, service,
                               notification_time_limits=NOTIFICATION_ALWAYS,
                               notification_level=CRIT,
                               flaps_config=juggler_sdk.FlapOptions(stable=180, critical=300, boost=0)):
    return juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': get_solomon_lag_checks(),
        },
        notifications=get_notifications_config(notification_level, notification_time_limits),
        flaps_config=flaps_config,
    )


def get_no_successful_sync_cycles(solomon_cluster):
    number_of_shards = solomon_cluster['number_of_shards']

    def build_shard_sensor(shard_id: int) -> str:
        # TODO(ismagilas): enpoint -> endpoint, first needs fix everywhere else
        return f"let sensors_shard_{shard_id} = group_lines('sum', \
                {{service='{solomon_cluster['service']}', \
                sensor='service_controller.enpoint_set_manager_factory_shard_id_{shard_id}.successful_sync_cycles', \
                host='cluster', \
                cluster='{solomon_cluster['name']}'}});"

    yp_cluster = solomon_cluster['name'].split('_yp')[0].replace('_', '-')
    notification_time_limits = NOTIFICATION_WORK_TIME if (yp_cluster == 'sas-test' or yp_cluster == 'man-pre') else NOTIFICATION_ALWAYS

    host=f"{yp_cluster}.{JUGGLER_HOST}"
    service='no-successful-sync-cycles-on-shards'

    solomon_alert = get_solomon_expression_alert(
        name=f"{service}-{yp_cluster}",
        title_name=f"No successful sync cycles on shards for {yp_cluster}",
        program='\n'.join((build_shard_sensor(i) for i in range(number_of_shards))),
        check_expression=' || '.join(f'sum(sensors_shard_{i}) == 0' for i in range(number_of_shards)),
        annotations={
            "service": service,
            "host": host,
            "alert_description": f"No successful sync cycles on shards for cluster {yp_cluster}"
        },
        period_ms=milliseconds(seconds=300),
        delay_s=30,
    )
    juggler_check = get_juggler_check_with_yp_downtime(host, service, yp_cluster,
                                                       notification_time_limits=notification_time_limits,
                                                       ttl=900,
                                                       flaps_config=juggler_sdk.FlapOptions(stable=900, critical=2700, boost=0))
    return solomon_alert, juggler_check


def get_unexpected_number_of_shards(solomon_cluster):
    number_of_shards = solomon_cluster['number_of_shards']

    def build_shard_sensor(shard_id_to_check: int) -> str:
        return f"let sensor = group_lines('sum', \
                {{service='{solomon_cluster['service']}', \
                sensor='service_controller.enpoint_set_manager_factory_shard_id_{shard_id_to_check}.lock_acquired', \
                host='cluster', \
                cluster='{solomon_cluster['name']}'}});"

    yp_cluster = solomon_cluster['name'].split('_yp')[0].replace('_', '-')
    notification_time_limits = NOTIFICATION_WORK_TIME if (yp_cluster == 'sas-test' or yp_cluster == 'man-pre') else NOTIFICATION_ALWAYS

    host=f"{yp_cluster}.{JUGGLER_HOST}"
    service='unexpected-number-of-shards'

    solomon_alert = get_solomon_expression_alert(
        name=f"{service}-{yp_cluster}",
        title_name=f"Unexpected number of shards for {yp_cluster}",
        program=build_shard_sensor(number_of_shards),
        check_expression='sum(sensor) != 0',
        annotations={
            "service": service,
            "host": host,
            "alert_description": f"Unexpected number of shards for {yp_cluster}"
        },
        period_ms=milliseconds(seconds=300),
        delay_s=30,
        resolved_empty_policy="RESOLVED_EMPTY_OK",
        no_points_policy="NO_POINTS_OK"
    )
    juggler_check = get_juggler_check_with_yp_downtime(host, service, yp_cluster,
                                                       notification_time_limits=notification_time_limits,
                                                       ttl=900,
                                                       flaps_config=juggler_sdk.FlapOptions(stable=900, critical=2700, boost=0))
    return solomon_alert, juggler_check


def get_solomon_lag_check():
    return juggler_sdk.Check(
        host='solomon',
        service=get_juggler_service('solomon_lag'),
        namespace=JUGGLER_NAMESPACE,
        aggregator='logic_or',
        children=[
            juggler_sdk.Child(
                host='solomon_alerting_project_lag',
                service='service_controller',
                group_type='HOST',
                instance=''
            ),
        ],
    )


SOLOMON_CLUSTERS = [
    {
        'name': 'sas_test_yp',
        'service': 'sas_test_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'man_pre_yp',
        'service': 'man_pre_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'sas_yp',
        'service': 'sas_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'vla_yp',
        'service': 'vla_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'myt_yp',
        'service': 'msk_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'iva_yp',
        'service': 'iva_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
    {
        'name': 'xdc_yp',
        'service': 'xdc_yp',
        'number_of_shards': 5,
        'alerts': [
            get_no_successful_sync_cycles,
            get_unexpected_number_of_shards
        ],
    },
]


def update_juggler_check(juggler_client, check):
    if isinstance(check, juggler_sdk.CheckFilter):
        result = juggler_client.remove_checks([check]).results[0]
        logging.info('Removed juggler check {}:{}: {}'.format(check.host, check.service, 'Success' if result.success else 'Failed'))
    else:
        diff = juggler_client.upsert_check(check).diff.to_dict()
        logging.info('Updated juggler check {}:{}'.format(check.service, check.service))
        logging.info('Diff:\n{}'.format(json.dumps(diff, indent=2)))


def update_alert(solomon_cluster, solomon_client, juggler_client, solomon_config, juggler_check):
    solomon_client.put_item('alerts', solomon_config['id'], solomon_config)
    juggler_check.meta.setdefault('urls', []).extend(get_standard_urls(solomon_config['id'], solomon_cluster))
    update_juggler_check(juggler_client, juggler_check)


def update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster):
    for get_alert_func in solomon_cluster.get('alerts', []):
        update_alert(solomon_cluster, solomon_client, juggler_client, *get_alert_func(solomon_cluster))


def update_alerts(solomon_client, juggler_client):
    update_juggler_check(juggler_client, get_solomon_lag_check())

    for solomon_cluster in SOLOMON_CLUSTERS:
        update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster)


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--solomon-token', required=True, help='solomon token. https://solomon.yandex-team.ru/api/internal/auth')
    parser.add_argument('--juggler-token', required=True, help='juggler token. https://juggler.yandex-team.ru/')
    parser.add_argument('--verbose', action="store_true", help='Enable verbose mode')
    return parser.parse_args(argv)


def main(argv):
    args = parse_args(argv)
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO))

    solomon_client = SolomonApi(SOLOMON_PROJECT_ID, args.solomon_token)
    with juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=args.juggler_token, mark=MARK) as juggler_client:
        update_alerts(solomon_client, juggler_client)

if __name__ == '__main__':
    main(sys.argv[1:])
