import argparse
import json
import juggler_sdk
import logging
import sys

from infra.yp_service_discovery.monitoring.solomon.src.util.solomon_api import SolomonApi

SOLOMON_PROJECT_ID = 'service_controller_shard_master'

JUGGLER_HOST = 'service-controller-shard-master'
JUGGLER_NAMESPACE = 'yp.service_controller.shard_master'

WARN = 'warn'
CRIT = 'crit'

MARK = 'service-controller-shard-master'

LOGINS = {
    WARN: [
        '@svc_service_discovery:yp-service-discovery-duty',
    ],
    CRIT: [
        '@svc_service_discovery:yp-service-discovery-duty',
        'dns-monitoring',
    ],
}

PHONE_ESCALATION_LOGINS = {
    WARN: [
    ],
    CRIT: [
        '@svc_service_discovery:yp-service-discovery-duty',
        'ismagilas',
        'elshiko',
        'avitella',
    ],
}

NOTIFICATION_ALWAYS = {
    "day_start": 1,
    "day_end": 7,
    "time_start": 0,
    "time_end": 23,
}

NOTIFICATION_WORK_TIME = {
    "day_start": 1,
    "day_end": 5,
    "time_start": 10,
    "time_end": 21,
}


def milliseconds(seconds=0, minutes=0):
    return seconds * 1000 + minutes * 60 * 1000


def create_selectors(**kwargs):
    def create_selector(key, value):
        if value.startswith('!'):
            return '{}!="{}"'.format(key, value[1:])
        return '{}="{}"'.format(key, value)

    selectors = ', '.join(map(lambda kv: create_selector(*kv), kwargs.items()))
    return '{' + selectors + '}'


def notification(status, logins, notify_methods, time_limits=NOTIFICATION_ALWAYS):
    template_kwargs = {
        "status": status,
        "login": logins,
        "method": notify_methods,
        "day_start": time_limits["day_start"],
        "day_end": time_limits["day_end"],
        "time_start": time_limits["time_start"],
        "time_end": time_limits["time_end"],
    }

    if (isinstance(status, str) and status == "CRIT") or (isinstance(status, dict) and status["to"] == "CRIT"):
        template_kwargs["repeat"] = 1800  # 30 min

    return juggler_sdk.NotificationOptions(
        template_name="on_status_change",
        template_kwargs=template_kwargs,
    )


def get_notifications_config(level=CRIT, time_limits=NOTIFICATION_ALWAYS):
    logins = LOGINS[level]
    escalation_logins = PHONE_ESCALATION_LOGINS[level]

    result = []
    if logins:
        result.extend([
            notification(
                status={'from': 'OK', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ],
                time_limits=time_limits
            ),
            notification(
                status={'from': 'WARN', 'to': 'CRIT'},
                logins=logins,
                notify_methods=[
                    'sms',
                    'telegram',
                ],
                time_limits=time_limits
            ),
            notification(
                status={'from': 'CRIT', 'to': 'OK'},
                logins=logins,
                notify_methods=[
                    'telegram',
                ],
                time_limits=time_limits
            ),
        ])

    if escalation_logins:
        result.extend([
            juggler_sdk.NotificationOptions(
                template_name='phone_escalation',
                template_kwargs={
                    'delay': 60,
                    'logins': escalation_logins,
                    'day_start': time_limits['day_start'],
                    'day_end': time_limits['day_end'],
                    'time_start': time_limits['time_start'],
                    'time_end': time_limits['time_end'],
                }
            ),
        ])

    return result


def get_juggler_service(service_id):
    return '.'.join(['yp', 'service_controller', 'shard_master', service_id])


def get_yp_downtime_tags(yp_cluster):
    return [
        "yp-downtime-9b8ee2a2b559b3afff751623e5e79546",
        "yp-{}-downtime-9b8ee2a2b559b3afff751623e5e79546".format(yp_cluster),
    ]


def get_solomon_lag_checks():
    return ['solomon:{}'.format(get_juggler_service('solomon_lag'))]


def get_yp_unreach_checks(yp_cluster):
    return ['yp-{}.yp.yandex.net:yp.infra.auto_downtime_'.format(yp_cluster)]


def get_yp_standard_unreach_checks(yp_cluster):
    return get_yp_unreach_checks(yp_cluster) + get_solomon_lag_checks()


def get_solomon_alert_url(alert_id):
    return {
        'url': 'https://solomon.yandex-team.ru/admin/projects/{}/alerts/{}'.format(SOLOMON_PROJECT_ID, alert_id),
        'title': 'Alert on solomon',
    }


def get_nanny_service_url(yp_cluster):
    return {
        'url': 'https://nanny.yandex-team.ru/ui/#/services/catalog/{}_yp_service_controller_shard_master'.format(yp_cluster.replace('-', '_')),
        'title': 'Nanny service',
        'type': 'nanny',
    }


def get_source_url():
    return {
        'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/infra/service_controller_shard_master/monitoring/alerts',
        'title': 'Alerts generation source code',
    }


def get_standard_urls(alert_id, solomon_cluster):
    yp_cluster = solomon_cluster['name'].split('prod-', 1)[1]
    return [
        get_solomon_alert_url(alert_id),
        get_nanny_service_url(yp_cluster),
        get_source_url(),
    ]


def get_juggler_check_with_yp_downtime(host, service, yp_cluster,
                                       notification_time_limits=NOTIFICATION_ALWAYS,
                                       notification_level=CRIT,
                                       ttl=600,
                                       flaps_config=juggler_sdk.FlapOptions(stable=60, critical=300, boost=0)):
    return juggler_sdk.Check(
        host=host,
        service=service,
        namespace=JUGGLER_NAMESPACE,
        tags=get_yp_downtime_tags(yp_cluster),
        aggregator='logic_or',
        aggregator_kwargs={
            'unreach_checks': get_yp_standard_unreach_checks(yp_cluster),
        },
        ttl=ttl,
        notifications=get_notifications_config(notification_level, notification_time_limits),
        flaps_config=flaps_config,
    )


def get_no_successful_resharding_cycles(solomon_cluster):
    yp_cluster = solomon_cluster['name'].split('prod-', 1)[1]
    service_id = 'no-successful-resharding-cycles-{}'.format(yp_cluster)
    notification_time_limits = NOTIFICATION_WORK_TIME if (yp_cluster == 'sas-test' or yp_cluster == 'man-pre') else NOTIFICATION_ALWAYS
    sensor_name = 'shard_master.controller_shard_master.successful_resharding_cycles'

    return {
        'id': service_id,
        'projectId': SOLOMON_PROJECT_ID,
        'name': 'No successful resharding cycles for YP-{}'.format(yp_cluster.upper()),
        'notificationChannels': [
            'juggler',
        ],
        'type': {
            'threshold': {
                'selectors': '{' + 'cluster="{}", service="main", host="all", service_name="service_controller", sensor="{}"'.format(solomon_cluster['name'], sensor_name) + '}',
                'timeAggregation': 'SUM',
                'predicate': 'EQ',
                'threshold': 0,
            }
        },
        'periodMillis': milliseconds(seconds=90),
        'delaySeconds': 0,
        'annotations': {
            'service': service_id,
            'host': JUGGLER_HOST,
        },
    }, get_juggler_check_with_yp_downtime(JUGGLER_HOST, service_id, yp_cluster,
                                          notification_time_limits=notification_time_limits)


def get_solomon_lag_check():
    return juggler_sdk.Check(
        host='solomon',
        service=get_juggler_service('solomon_lag'),
        namespace=JUGGLER_NAMESPACE,
        aggregator='logic_or',
        children=[
            juggler_sdk.Child(
                host='solomon_alerting_project_lag',
                service='service_controller_shard_master',
                group_type='HOST',
                instance=''
            ),
        ],
    )


SOLOMON_CLUSTERS = [
    {
        'name': 'prod-sas-test',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-man-pre',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-iva',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-myt',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-vla',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-sas',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
    {
        'name': 'prod-xdc',
        'alerts': [
            get_no_successful_resharding_cycles
        ],
    },
]


def update_juggler_check(juggler_client, check):
    if isinstance(check, juggler_sdk.CheckFilter):
        result = juggler_client.remove_checks([check]).results[0]
        logging.info('Removed juggler check {}:{}: {}'.format(check.host, check.service, 'Success' if result.success else 'Failed'))
    else:
        diff = juggler_client.upsert_check(check).diff.to_dict()
        logging.info('Updated juggler check {}:{}'.format(check.service, check.service))
        logging.info('Diff:\n{}'.format(json.dumps(diff, indent=2)))


def update_alert(solomon_cluster, solomon_client, juggler_client, solomon_config, juggler_check):
    solomon_client.put_item('alerts', solomon_config['id'], solomon_config)
    juggler_check.meta.setdefault('urls', []).extend(get_standard_urls(solomon_config['id'], solomon_cluster))
    update_juggler_check(juggler_client, juggler_check)


def update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster):
    for get_alert_func in solomon_cluster.get('alerts', []):
        update_alert(solomon_cluster, solomon_client, juggler_client, *get_alert_func(solomon_cluster))


def update_alerts(solomon_client, juggler_client):
    update_juggler_check(juggler_client, get_solomon_lag_check())

    for solomon_cluster in SOLOMON_CLUSTERS:
        update_alerts_for_cluster(solomon_client, juggler_client, solomon_cluster)


def parse_args(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--solomon-token', required=True, help='solomon token. https://solomon.yandex-team.ru/api/internal/auth')
    parser.add_argument('--juggler-token', required=True, help='juggler token. https://juggler.yandex-team.ru/')
    parser.add_argument('--verbose', action="store_true", help='Enable verbose mode')
    return parser.parse_args(argv)


def main(argv):
    args = parse_args(argv)
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO))

    solomon_client = SolomonApi(SOLOMON_PROJECT_ID, args.solomon_token)
    with juggler_sdk.JugglerApi('http://juggler-api.search.yandex.net', oauth_token=args.juggler_token, mark=MARK) as juggler_client:
        update_alerts(solomon_client, juggler_client)

if __name__ == '__main__':
    main(sys.argv[1:])
