from gevent import subprocess
from retry import retry
import logging

import utils


def build_shard(build_cmd, output_dir, generation, shard, tier):
    utils.execute([
        '/bin/sh', build_cmd,
        '-o', output_dir,
        '-g', generation,
        '-s', shard,
        '-t', tier,
    ], raise_on_error=True)


def sky_get(resource_url, dst_dir, timeout, max_dl_speed=None, copier_opts=None):
    args = [
        'sky', 'get',
        '-uw',
        '-t', str(timeout),
    ]
    if max_dl_speed:
        args += ['--max-dl-speed={}'.format(max_dl_speed)]
    if copier_opts:
        args += ['--opts={}'.format(copier_opts)]
    args += [
        '-d', dst_dir,
        resource_url,
    ]

    return utils.execute(args) == 0


@retry(tries=120, delay=60)
def sky_share(src_dir):
    # remove unworking option
    # cmd = ['sky share --opts "direct: 1" -d "{}" $(cd "{}"; find -type f) || exit 1'.format(src_dir, src_dir)]
    cmd = ['sky share -d "{}" $(cd "{}"; find -type f) || exit 1'.format(src_dir, src_dir)]
    logging.info('Call %s', str(cmd))

    p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    out, err = p.communicate()
    return_code = p.returncode

    if return_code != 0:
        raise SubprocessException(cmd)

    return out.strip()


def configure_shard(shard_dir, shard_id):
    utils.execute(['./iss_shards', 'configure', '--id', shard_id, shard_dir], raise_on_error=True)


@retry(tries=3, delay=1, backoff=2)
def register_shard(shard_dir):
    utils.execute(['./iss_shards', 'register', shard_dir], raise_on_error=True)


def find_rbtorrent(shard_id):
    cmd = ['./iss_shards', 'info', shard_id]
    logging.info('Call %s', str(cmd))

    out, err = subprocess.Popen(
        cmd,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE).communicate()

    urls = [l.strip() for l in out.splitlines() if l.startswith('rbtorrent')]
    return urls[0] if urls else None


class SubprocessException(Exception):
    def __init__(self, cmd):
        super(SubprocessException, self).__init__('[{}] failed'.format(cmd[0]))
