import logging
from datetime import datetime, timedelta
import gevent
import gevent.pool
import pprint
import requests

from enums import Status, Policy
from policy import assign_builders, InstanceState


class PeerSet(gevent.Greenlet):
    def __init__(self, shardmap, me):
        super(PeerSet, self).__init__()

        self._peers = set()
        self._shard_map = {}
        self._instance = me
        self._interested_shards = set()
        self._policy = _find_my_policy(shardmap, me)
        self._static_rbtorrents = {}

        for shard_id, shard_obj in shardmap.iteritems():
            if 'rbtorrent' in shard_obj:
                self._static_rbtorrents[shard_id] = shard_obj['rbtorrent']

            shard_peers = set(peer['instance'] for peer in (shard_obj['instances']))
            if self._instance not in shard_peers:
                continue

            self._interested_shards.add(shard_id)

            if self._policy in (Policy.BUILDER, Policy.LEECHER):
                self._peers |= {self._instance}
                self._shard_map[shard_id] = [{
                                                 'instance': self._instance,
                                                 'status': Status.IDLE,
                                                 'timestamp': datetime.now(),
                                                 'policy': self._policy
                                            }]
            else:
                self._peers |= shard_peers
                self._shard_map[shard_id] = [{
                                                 'instance': peer['instance'],
                                                 'status': Status.IDLE,
                                                 'timestamp': datetime.now(),
                                                 'policy': peer['policy']
                                            } for peer in shard_obj['instances'] if peer['policy'] != Policy.LEECHER]

    def i_am_the_builder(self, shard_id):
        return self._instance in self._assign_builders(shard_id)

    def i_am_the_downloader(self, shard_id):
        return self._instance in self._assign_downloaders(shard_id)

    @property
    def shards(self):
        return self._shard_map.keys()

    @property
    def peers(self):
        return self._peers

    @property
    def interested_shards(self):
        return self._interested_shards

    def find_static_rbtorrent(self, shard_id):
        return self._static_rbtorrents.get(shard_id, None)

    def _assign_builders(self, shard):
        state = {}
        for shard_id in self.interested_shards:
            instances = self._shard_map[shard_id]
            state[shard_id] = []
            for instance in instances:
                status = instance['status'] if not _long_ago(instance['timestamp']) else Status.NONE
                state[shard_id].append(InstanceState(instance['instance'], status, 0.0, instance['policy']))

        _log.debug('FULL STATE:\n%s', pprint.pformat(state))
        return assign_builders(state).get(shard, set())

    def _assign_downloaders(self, shard_id):
        return set(peer['instance'] for peer in self._shard_map[shard_id] if peer['policy'] == Policy.LEECHER)

    def _run(self):
        while True:
            try:
                pool = gevent.pool.Pool(10)
                for peer, status in pool.imap_unordered(_get_peer_status, self._peers):
                    for shard_id, shard_status in status.iteritems():
                        self._update(shard_id, peer, shard_status)
                pool.join()
            except Exception:
                _log.exception('Exception in PeerSet loop')

            gevent.sleep(15)

    def _update(self, shard_id, peer, status):
        if shard_id not in self._shard_map:
            return

        _log.debug('shard status report %s/%s: [%s]', shard_id, peer, status)
        for p in self._shard_map[shard_id]:
            if p['instance'] == peer:
                p['status'] = status['status']
                p['timestamp'] = datetime.now()


def _get_peer_status(peer):
    try:
        return peer, requests.get('http://{}/status'.format(peer), timeout=10).json()['shards']
    except requests.RequestException:
        _log.debug('%s is unavailable', peer)
        return peer, {}


def _find_my_policy(shardmap, me):
    for shard_obj in shardmap.itervalues():
        for peer in shard_obj['instances']:
            if peer['instance'] == me:
                return peer['policy']


def _long_ago(timestamp):
    return datetime.now() - timestamp > timedelta(minutes=10)


_log = logging.getLogger(__name__)
