# coding=utf-8
from collections import namedtuple
from enums import Status, Policy


InstanceState = namedtuple('InstanceState', ['instance', 'status', 'progress', 'policy'])


def assign_builders(state):
    stats = _Stats(state)

    assigned = {shard: set() for shard in state}

    # [rule 0] если живых инстансов мало (меньше 80% от популяции), то не делать ничего
    if len(stats.alive_instances) <= 0.8 * len(stats.all_instances):
        return assigned

    # [rule 1] мертвые инстансы заменить живыми в порядке, заданном картой реплик
    for shard, peers in state.iteritems():
        assigned[shard].update(assign_default_peer(peers))

    # [rule 2] заменить тех, кто сделал много попыток: больше sigma и при этом больше 5
    # (вычислить среднее и отклонение для количества попыток построения –> build_attempt: avg, sigma)
    # TODO: implementation

    # [rule 3] если непостроивших осталось столько, что хватит резерва, то включить резерв
    if len(stats.free_instances) >= len(stats.undone_shards):
        for shard in stats.undone_shards:
            assigned[shard].update(assign_free_peer(state[shard]))

    # [rule 4] если недостигнувших двух целей столько, что можно заменить их половиной резерва, то заменить
    elif False:
        # TODO: implementation
        pass

    return assigned


def assign_default_peer(peers):
    for peer in peers:
        if operational(peer) and peer.policy in (Policy.BUILDER, Policy.BUILDER_RESERVE, Policy.BUILDER_SPECIAL):
            return {peer.instance}
    return set()


def assign_free_peer(peers):
    for peer in peers:
        if peer.status == Status.IDLE and peer.policy in (Policy.BUILDER_RESERVE, Policy.BUILDER_SPECIAL):
            return {peer.instance}
    return set()


def operational(peer):
    return peer.status not in (Status.NONE, Status.FAILURE)


class _Stats(object):
    def __init__(self, state):
        """
        :type state: dict[str, list[InstanceState]]
        """
        self._all_shards = set(state.keys())
        self._instances = set()
        self._alives = set()
        self._free = set()
        self._done_shards = set()

        for shard, peers in state.iteritems():
            for peer in peers:
                self._instances.add(peer.instance)
                if peer.status != Status.NONE:
                    self._alives.add(peer.instance)
                if peer.status == Status.IDLE and peer.policy in (Policy.BUILDER_RESERVE, Policy.BUILDER_SPECIAL):
                    self._free.add(peer.instance)
                if peer.status == Status.DONE:
                    self._done_shards.add(shard)

    @property
    def alive_instances(self):
        return self._alives

    @property
    def all_instances(self):
        return self._instances

    @property
    def free_instances(self):
        return self._free

    @property
    def undone_shards(self):
        return self._all_shards - self._done_shards


def test_no_operations_if_not_enough_alives():
    state = {
        'shard1': [
            InstanceState('i1', Status.BUILD, 0, Policy.BUILDER),
            InstanceState('i2', Status.NONE, 0, Policy.BUILDER_RESERVE),
        ],
        'shard2': [
            InstanceState('i11', Status.NONE, 0, Policy.BUILDER),
            InstanceState('i12', Status.IDLE, 0, Policy.BUILDER_RESERVE),
        ]
    }
    assert assign_builders(state) == {'shard1': set(), 'shard2': set()}


def test_x_basic_policy():
    state = {
        'shard1': [
            InstanceState('i1', Status.NONE, 0.5, Policy.BUILDER),
            InstanceState('i2', Status.IDLE, 0, Policy.BUILDER_RESERVE),
            InstanceState('i3', Status.IDLE, 0, Policy.BUILDER_RESERVE),
            InstanceState('i4', Status.IDLE, 0, Policy.BUILDER_RESERVE),
            InstanceState('i5', Status.IDLE, 0, Policy.BUILDER_RESERVE),
        ],
        'shard2': [
            InstanceState('i11', Status.FAILURE, 0.75, Policy.BUILDER),
            InstanceState('i12', Status.IDLE, 0, Policy.BUILDER_RESERVE),
        ],
        'shard3': [
            InstanceState('i31', Status.IDLE, 0.75, Policy.BUILDER),
        ],
        'shard4': [
            InstanceState('i41', Status.IDLE, 0.75, Policy.LEECHER),
        ],
    }
    assert assign_builders(state) == {'shard1': {'i2'}, 'shard2': {'i12'}, 'shard3': {'i31'}, 'shard4': set()}


def test_x_finishing_policy():
    state = {
        'shard1': [
            InstanceState('i1', Status.BUILD, 0.5, Policy.BUILDER),
            InstanceState('i2', Status.IDLE, 0.0, Policy.BUILDER_RESERVE),
        ],
        'shard2': [
            InstanceState('i11', Status.BUILD, 0.75, Policy.BUILDER_RESERVE),
            InstanceState('i12', Status.IDLE, 0.0, Policy.BUILDER_RESERVE),
        ]
    }
    assert assign_builders(state) == {'shard1': {'i1', 'i2'}, 'shard2': {'i11', 'i12'}}
