from collections import namedtuple
import gevent
import logging

import external
from peers import PeerSet, Status
from server import serve_forever
from storage import Storage


class ShardTool(object):
    def __init__(self, replicamap, build_cmd, working_root, release_root, me, max_dl_speed, copier_opts):
        self._storage = Storage(working_root, release_root, replicamap['config']['slot_size_gb'] * Gb)
        self._peers = PeerSet(replicamap['shards'], me)
        self._shards = {shard: {'status': Status.IDLE, 'build_attempt': 0, 'download_attempt': 0}
                        for shard in self._peers.shards}

        # build options
        self._build_cmd = build_cmd
        self._generation = replicamap['config']['generation']
        self._tier = replicamap['config']['tier']

        # download options
        self._max_dl_speed = max_dl_speed
        self._copier_opts = copier_opts

        self._meta = replicamap.get('meta', {})

    def run(self, port):
        self._remove_unneeded_shards()

        gevent.spawn(serve_forever, self._get_status, ('::', port))
        self._peers.start()

        while True:
            # noinspection PyBroadException
            try:
                self._step()
            except Exception:
                logging.exception('Unhandled exception in the main loop')

            gevent.sleep(5)

    def _step(self):
        shard = self._find_shard_to_download()
        if shard:
            self._download(shard.id, shard.rbtorrent)
        else:
            logging.info('nothing to download')
            shard = self._find_shard_to_build()
            if shard:
                try:
                    self._build(shard)
                except RuntimeError:
                    if self._shards[shard]['build_attempt'] > 3:
                        self._set_status(shard, Status.FAILURE)
                    else:
                        self._set_status(shard, Status.IDLE)
                    raise
            else:
                logging.info('nothing to build')

    def _find_shard_to_download(self):
        for shard_id in self._peers.shards:
            if self._peers.i_am_the_downloader(shard_id) and self._shards[shard_id]['status'] != Status.DONE:
                rbtorrent = self._peers.find_static_rbtorrent(shard_id) or external.find_rbtorrent(shard_id)

                if rbtorrent:
                    return RemoteShard(rbtorrent=rbtorrent, id=shard_id)

        return None

    def _find_shard_to_build(self):
        for shard_id in self._shards:
            if self._peers.i_am_the_builder(shard_id) and self._shards[shard_id]['status'] != Status.DONE:
                return shard_id
        return None

    def _download(self, shard_id, rbtorrent):
        if self._storage.is_ready(shard_id):
            self._set_status(shard_id, Status.DONE)

        self._set_status(shard_id, Status.DOWNLOAD)
        shard_dir = self._storage.get_slot(shard_id, build=False)

        if external.sky_get(rbtorrent, shard_dir, 60000, self._max_dl_speed, self._copier_opts):
            self._storage.set_ready(shard_id, build=False)
            self._set_status(shard_id, Status.DONE)
        else:
            self._set_status(shard_id, Status.IDLE)

    def _build(self, shard_id):
        if self._storage.is_ready(shard_id):
            self._set_status(shard_id, Status.DONE)
            return

        self._set_status(shard_id, Status.BUILD)
        build_dir = self._storage.get_slot(shard_id, build=True)

        external.build_shard(
            self._build_cmd,
            build_dir,
            self._generation,
            '-'.join(shard_id.split('-')[2:4]),  # take from replicamap
            self._tier,
        )
        external.configure_shard(build_dir, shard_id)

        self._set_status(shard_id, Status.SHARE)
        rbtorrent_in_share = self._storage.share(shard_id)

        self._storage.set_ready(shard_id, build=True)
        rbtorrent_in_tracker = external.find_rbtorrent(shard_id)
        if rbtorrent_in_tracker is None:
            external.register_shard(build_dir)
        else:
            assert rbtorrent_in_tracker == rbtorrent_in_share

        self._set_status(shard_id, Status.DONE)

    def _set_status(self, shard_id, status):
        logging.info('Status change: [%s] %s -> %s', shard_id, self._shards[shard_id]['status'], status)
        self._shards[shard_id]['status'] = status

        if status == Status.BUILD:
            self._shards[shard_id]['build_attempt'] += 1

        if status == Status.DOWNLOAD:
            self._shards[shard_id]['download_attempt'] += 1

    def _get_status(self):
        return {
            'shards': {shard: data for shard, data in self._shards.iteritems() if shard in self._peers.interested_shards},
            'meta': self._meta,
        }

    def _remove_unneeded_shards(self):
        for shard in self._storage.resources:
            if shard not in self._shards:
                self._storage.rm_slot(shard)


Gb = float(1024 ** 3)
RemoteShard = namedtuple('RemoteShard', ['id', 'rbtorrent'])
