import logging
import os

import external
import utils


class Storage(object):
    def __init__(self, working_root, release_root, slot_size):
        self._working_root = os.path.abspath(working_root)
        self._release_root = os.path.abspath(release_root)
        self._slot_size = slot_size
        utils.ensure_dir(os.path.join(self._working_root, 'build'))
        utils.ensure_dir(os.path.join(self._working_root, 'download'))
        utils.ensure_dir(os.path.join(self._working_root, 'ready'))

    @property
    def allocated(self):
        return self._slot_size * len(self._ls())

    @property
    def used(self):
        return utils.du(self._working_root, exclude_dotfiles=False)

    @property
    def free(self):
        return utils.free_space(self._working_root)

    @property
    def full(self):
        return self.allocated + self._slot_size > self.free + self.used

    @property
    def resources(self):
        return set(
            os.listdir(os.path.join(self._working_root, 'build')) +
            os.listdir(os.path.join(self._working_root, 'download'))
        )

    def get_slot(self, resource_id, build=False):
        path = self._slot_path(resource_id, build)
        if not os.path.exists(path):
            if self.full and (self.free / Gb) < 100.0:
                raise RuntimeError(
                    'Not enough disk space to acquire shards: {} Gb used, {} Gb allocated, {} Gb free.'.format(
                        self.used / Gb,
                        self.allocated / Gb,
                        self.free / Gb
                    )
                )

        utils.ensure_dir(path)
        return path

    def rm_slot(self, resource):
        utils.remove_dir(self._slot_path(resource, build=True))
        utils.remove_dir(self._slot_path(resource, build=False))
        utils.remove_file(self._ready_path(resource))
        utils.remove_file(self._ready_path(resource) + '.rbtorrent')

    def is_ready(self, resource):
        return bool(_read_rbtorrent(self._ready_path(resource)))

    def share(self, resource, build=True):
        rbtorrent = external.sky_share(self._slot_path(resource, build))
        _write_rbtorrent(self._ready_path(resource), rbtorrent)
        return rbtorrent

    def set_ready(self, resource, build):
        utils.remove_file(self._ready_path(resource))
        utils.create_symlink(self._slot_path(resource, build=build), self._ready_path(resource))
        utils.remove_file(self._release_path(resource))
        utils.create_symlink(self._slot_path(resource, build=build), self._release_path(resource))

    def _slot_path(self, resource, build):
        return os.path.join(self._working_root, 'build' if build else 'download', resource)

    def _ready_path(self, resource):
        return os.path.join(self._working_root, 'ready', resource)

    def _ls(self):
        return utils.ls(self._working_root + '/build') + utils.ls(self._working_root + '/download')

    def _release_path(self, resource):
        return os.path.join(self._release_root, resource)


def _write_rbtorrent(shard_dir, rbtorrent):
    with open(shard_dir + '.rbtorrent', 'w') as f:
        logging.info('Write rbtorrent [%s]', rbtorrent)
        f.write(rbtorrent + '\n')


def _read_rbtorrent(shard_dir):
    # noinspection PyBroadException
    try:
        with open(shard_dir + '.rbtorrent', 'r') as f:
            rbtorrent = f.read().strip()
            logging.info('Read rbtorrent [%s]', rbtorrent)
            return rbtorrent
    except Exception:
        return None


def test_operations():
    _mkdirs([
        './storage_root',
        './storage_root/build',
        './storage_root/build/p1',
        './storage_root/build/p2',
        './storage_root/build/p3',
        './storage_root/download',
        './storage_root/download/p1',
        './storage_root/download/p5',
        './storage_root/ready',
        './storage_root/ready/r1',
        './storage_root/ready/r2',
        './storage_root/ready/r5',
        './storage_root/p1',
        './storage_root/p5',
        './storage_root/p10',
    ])
    slot_size = 1024
    storage = Storage('./storage_root', slot_size)
    assert storage.allocated == 5 * slot_size

    slot = storage.get_slot('resource-1', build=True)
    with open(slot + '/data', 'a') as f:
        f.write('a' * 1024 * 1024)
    assert storage.allocated == 6 * slot_size
    assert 1024 * 1024 < storage.used < 1.2 * 1024 * 1024

    storage.rm_slot('resource-1')
    assert storage.allocated == 5 * slot_size
    assert storage.used < 1024 * 1024

    storage.get_slot('resource-2', build=False)
    assert {'p1', 'p2', 'p3', 'p5', 'resource-2'} == storage.resources
    storage.rm_slot('resource-2')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources
    storage.rm_slot('resource-2')
    storage.rm_slot('resource-2')
    storage.rm_slot('resource-2')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources

    slot = storage.get_slot('resource-3', build=True)
    with open(slot + '/data', 'a') as f:
        f.write('a' * 1024)
    storage.share('resource-3')
    storage.set_ready('resource-3', build=True)
    storage.rm_slot('resource-3')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources


def _mkdirs(dirs):
    map(utils.recreate_dir, dirs)


Gb = float(1024 ** 3)
