import argparse
import logging
import sys

from infra.shawshank.lib import master
from infra.shawshank.lib import tun64

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG, format='%(asctime)s %(message)s')


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--config", type=str, default=master.CONFIG_PATH, help="Path to file with tunnels description")
    parser.add_argument("-i", "--iface", type=str, default='veth', help="Determine interface to get local ip")
    parser.add_argument("-m", "--dont-check-mtn", action='store_false', dest="check_mtn", help="Don't check if local ip is in mtn")
    parser.add_argument("-n", "--no-shawshank-tunnels", action='store_true', dest="no_shawshank", help="Don't create shawshank tunnels")
    parser.add_argument("-t", "--tun64", action="store_true", dest="tun64_route", help="Configure default route for tun64 tunnel")
    parser.add_argument("--noporto", action="store_true", dest="noporto", help="Don't rely on porto")
    args = parser.parse_args()
    return args


def configure_shawshank_tunnels(args):
    container_spec = master.read_spec(args.config)
    if not container_spec:
        logging.critical("No spec for shawshank tunnels, exit")
        sys.exit(1)

    if not master.is_valid_spec(container_spec, args.noporto):
        logging.critical("Spec is not valid, exit")
        sys.exit(1)

    current_tunnels = master.form_current_tunnels_proto()

    if len(current_tunnels.tunnels) == 0:
        master.create_tunnels(container_spec, args.iface, args.check_mtn)
        master.check_and_add_addr(container_spec)
        master.check_default_routes(container_spec)
        return

    diff = master.calc_diff(current_tunnels, container_spec)
    if diff.get('old'):
        master.delete_tunnels(diff['old'])

    if diff.get('changed'):
        master.change_tunnels(*diff['changed'], iface_name=args.iface, check_if_in_mtn=args.check_mtn)

    if diff.get('new'):
        master.create_tunnels(diff['new'], args.iface, args.check_mtn)
    master.check_and_add_addr(container_spec)
    master.check_default_routes(container_spec)


def configure_tun64(bb_iface='veth', check_if_mtn=False):
    logging.info("Trying to configure tun64 iface")
    try:
        tun64_iface = tun64.create_tun64_iface(bb_iface, check_if_mtn)
        tun64.check_default_tun64_route(tun64_iface)
    except Exception as e:
        logging.crititcal("Could not configure tun64 iface: {}".format(e))
        sys.exit(1)


def main():
    args = parse_args()

    if not args.no_shawshank:
        configure_shawshank_tunnels(args)

    if args.tun64_route:
        configure_tun64(args.iface, args.check_mtn)


if __name__ == '__main__':
    main()
