import netaddr
import socket
import logging
import sys
from pyroute2 import IPRoute

from infra.shawshank.lib import netlink

DECAP_ADDR = '2a02:6b8:0:3400::aaaa'
DECAP_NETADDR = netaddr.IPAddress(DECAP_ADDR)
DECAP_MTU = 1450


def create_decap_iface():
    try:
        local_ip = netlink.get_local_bb_ip('veth')
    except IndexError:
        raise Exception('Could not get local backbone IP')
    return netlink.create_ip6_tun('veth', 4, local_ip, DECAP_ADDR, DECAP_MTU)


def get_decap_iface():
    link_id = None
    links = netlink.get_links()
    for link in links:
        remote_ip, remote_proto = netlink.get_remote_ip_and_proto(link)
        if remote_ip:
            remote_ip = netaddr.IPAddress(remote_ip)
            if DECAP_NETADDR == remote_ip:
                link_id = link.get('index')
                break
    return link_id


def get_addresses_of_decap_iface(iface_id):
    addresses = set()
    with IPRoute() as ipr:
        addresses = {a.get_attrs('IFA_ADDRESS')[0] for a in ipr.get_addr(family=socket.AF_INET, index=iface_id)}
    return addresses


def create_decap_rules(addresses, table_id=4010, priority=16000):
    with IPRoute() as ipr:
        ipr.flush_rules(family=socket.AF_INET)
        for a in addresses:
            try:
                ipr.rule(
                    'add',
                    src=a,
                    table=table_id,
                    priority=priority,
                    family=socket.AF_INET,
                )
            except:
                logging.warning("Could not add rule 'from {} lookup {} prio {}'".format(a, table_id, priority))
                sys.exit(1)


def create_route_for_decap(link_id, table_id=4010):
    with IPRoute() as ipr:
        ipr.flush_routes(family=socket.AF_INET, table=table_id)
        try:
            ipr.route(
                'add',
                table=table_id,
                oif=link_id,
                scope=253,
            )
        except:
            logging.warning("Could not add route for {} table_id".format(table_id))
            sys.exit(1)
