import logging
import socket
import requests
import sys
import netaddr
import json
import os
import time

from pyroute2 import IPRoute

from infra.shawshank.lib import netlink

TUN64_MAP_URL = 'http://noc-export.yandex.net/rt/map64.json'
TUN64_MAP_CACHE = '/tmp/map64.json'
TUN64_MAP_CACHE_TTL = 1800  # 30 min
TUN64_DECAP = '2a02:6b8:b010:a0ff::1'
TUN64_IFACE_NAME = 'ip_ext_tun0'
TUN64_MTU = 1400
MTN_BACKBONE_NET = netaddr.IPNetwork("2a02:6b8:c00::/40")


def get_tun64_map(url=TUN64_MAP_URL):
    try:
        # Read from cache
        if os.path.exists(TUN64_MAP_CACHE) and time.time() - os.path.getmtime(TUN64_MAP_CACHE) < TUN64_MAP_CACHE_TTL:
            with open(TUN64_MAP_CACHE, 'r') as f:
                return json.load(f)

        # if no cache or outdated get new one
        r = requests.get(url)
        if r.status_code == requests.codes.ok:
            ret = r.json()
            with open(TUN64_MAP_CACHE, 'w') as f:
                json.dump(ret, f)
            return ret
        else:
            logging.warning("Got {} status code while getting nat64 map".format(r.status_code))

    except Exception as e:
        logging.warning("Got an exception on getting tun64 map:\n{}".format(e))


def transponse_map64(tun64_map):
    r = {}
    for k, v in tun64_map.items():
        try:
            r[v['addr6']] = k
        except KeyError:
            pass
    return r


def create_tun64_iface(bb_iface, check_if_in_mtn, global_bb_address=None):
    exists_id = find_tun64_iface()
    if exists_id:
        logging.info("Already exists")
        return exists_id

    tun64_map = get_tun64_map()
    if not tun64_map:
        raise Exception('tun64 map is empty')

    tun64_map_t = transponse_map64(tun64_map)

    set_of_v6 = {netaddr.IPAddress(k) for k in tun64_map_t.keys()}
    if not global_bb_address:
        try:
            global_bb_address = netlink.get_local_bb_ip(bb_iface, check_if_in_mtn)
            logging.info('using {} address'.format(global_bb_address))
        except IndexError:
            raise Exception('Could not get local backbone ip address')

    if netaddr.IPAddress(global_bb_address) not in set_of_v6:
        raise Exception('Not found any v6 address to use as local address for tun64 tunnel')
    else:
        logging.info('{} in set of v6'.format(global_bb_address))

    link_id = netlink.create_ip6_tun(TUN64_IFACE_NAME, netlink.PROTO_IPIP6, global_bb_address, TUN64_DECAP, TUN64_MTU)
    with IPRoute() as ipr:
        addr = tun64_map_t.get(global_bb_address)
        logging.info('Adding {} to tunnel {}'.format(addr, TUN64_IFACE_NAME))
        ipr.addr('add', index=link_id, address=addr, mask=32)
    return link_id


def try_find_tun64_via_map():
    tun64_interface = None
    tun64_map = get_tun64_map()
    if not tun64_map:
        logging.critical("tun64 map is empty")
        return

    tun64_keys = {netaddr.IPAddress(ip) for ip in tun64_map.keys()}
    with IPRoute() as ipr:
        addresses = ipr.get_addr(family=socket.AF_INET)
    for address in addresses:
        if netaddr.IPAddress(address.get_attr('IFA_ADDRESS')) in tun64_keys:
            tun64_interface = address.get('index')
            break

    return tun64_interface


def find_tun64_iface():
    tun64_interface = netlink.get_iface_id(TUN64_IFACE_NAME)
    if not tun64_interface:
        logging.warning("Could not find {} iface".format(TUN64_IFACE_NAME))
        logging.info("Trying to find iface via tun64 map")
        tun64_interface = try_find_tun64_via_map()

    logging.info("tun64 iface: {}".format(tun64_interface))
    return tun64_interface


def check_default_tun64_route(oif_id):
    try:
        netlink.check_default_route(socket.AF_INET, oif_id)
    except Exception as e:
        logging.critical("Could not set default route to tun64 iface: {}".format(e))
        sys.exit(1)
