import contextlib
import logging
import os

from gevent import GreenletExit

from ..logger import SmartLoggerAdapter
from ..utils import Path

from ...rpc.errors import CallFail

from api.logger import SkynetLoggingHandler
from api.copier import errors


class PidLoggerAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        pid = self.extra['pid']
        ppid = self.extra['ppid']

        msg = '[%d:%d]  %s' % (ppid, pid, msg)

        return msg, kwargs

    def getChild(self, name):  # noqa
        name = '%-10s' % (name, )
        child = super(PidLoggerAdapter, self).getChild(name)
        child.logger.name = '%-24s' % ('.'.join(part.strip() for part in child.name.split('.')), )
        return child


def setup_logger(name):
    log = logging.getLogger('')
    handler = SkynetLoggingHandler(app='skybone', filename='skybone-%s.log' % (name, ))
    handler.setLevel(logging.DEBUG)
    log.setLevel(logging.DEBUG)
    log.addHandler(handler)

    return PidLoggerAdapter(log, {'pid': os.getpid(), 'ppid': os.getppid()})


def path_locker_proxy(job):
    for msg, payload in job:
        if msg == 'locked':
            paths = payload
            yield [Path(p) for p in paths]
        elif msg == 'done':
            return


@contextlib.contextmanager
def path_locker(master_cli, paths):
    job = master_cli.call('lock_files', paths)
    assert job.next() == ['start', None]

    yield path_locker_proxy(job)

    job.send(True)
    job.wait()


@contextlib.contextmanager
def dl_helper(master_cli, resid, dest, grn):
    dest_inode = dest.stat().ino
    job = master_cli.call('dl_helper', resid, dest.strpath, dest_inode)

    class DlHelper(object):
        def _get_response(self, cmd, expected):
            job.send(cmd)
            try:
                msg = job.next()
            except StopIteration:
                raise errors.CopierError('Copier didnt responded to "%s" command' % (cmd[0], ))
            except CallFail as ex:
                raise ex.args[3]

            if msg[0] != expected:
                raise errors.CopierError('Copier responded to "%s" command not expected result "%s"' % (
                    cmd[0], msg[0])
                )

            return msg[1:]

        def get_props(self):
            props = self._get_response(('get_props', ), 'props')[0]
            return props

        def lock_resource(self):
            msg = self._get_response(('lock_resource', False), 'resource_locked')
            assert msg[0] == resid

        def lock_resource_or_get_head(self):
            msg = self._get_response(('lock_resource', True), 'resource_locked')
            assert msg[0] == resid
            return msg[1]

        def check_dest(self, dest):
            inode = dest.stat().ino
            dev = dest.stat().dev
            msg = self._get_response(('check_dest', dest.strpath, inode, dev), 'dest_checked')
            assert msg[0] == dest.strpath, 'Unable to convert destination path (%s) thru porto/lxc' % (dest.strpath, )
            return msg[1]  # dest_in_dom0

        def set_world_sock(self, sock):
            msg = self._get_response(('set_sock', sock), 'sock_set')
            assert list(msg[0]) == list(sock), 'Invalid sock response %r' % (sock, )

        def set_head(self, head):
            job.send(('set_head', head))
            msg = job.next()
            assert msg[0]

        def get_alternatives(self):
            msg = self._get_response(('get_file_alternatives', ), 'alternatives')
            assert isinstance(msg[0], dict)
            return msg[0]

        def lock_files(self):
            self._get_response(('lock_files', ), 'paths_locked')
            return True

        def get_head_from_cache(self, resid):
            msg = self._get_response(('get_head', resid), 'head')
            assert msg[0] is None or isinstance(msg[0], dict)
            return msg[0]

        def _check_paths_cv_info(self, pathtuple):
            if pathtuple[0] != pathtuple[1]:
                return '%s (%s)' % (pathtuple[1], pathtuple[0])
            else:
                return pathtuple[0]

        def check_paths(self, files):
            msg = self._get_response(('check_paths', files), 'paths_checked')

            if msg[0]:
                if len(msg[0]) < 10:
                    filelist = ', '.join((self._check_paths_cv_info(pathtuple) for pathtuple in msg[0]))
                else:
                    filelist = '%s and other %d files' % (self._check_paths_cv_info(msg[0][0]), len(msg[0]) - 1)

                raise errors.FilesystemError(
                    'skybone daemon has no access to %s (either file(s) have invalid perms or not exist)' % (
                        filelist,
                    )
                )

        def store_resource(self, resource_dict):
            self._get_response(('store_resource', resource_dict), 'resource_stored')
            return

    yield DlHelper()
    job.send(('done', True))
    job.wait()


class MasterDisconnect(GreenletExit):
    pass


@contextlib.contextmanager
def master_mon(master_cli, subproc):
    import gevent

    job = master_cli.call('mon', subproc)
    grn = gevent.getcurrent()

    def _mon():
        try:
            job.next()
        except gevent.GreenletExit:
            raise
        except:
            grn.kill(MasterDisconnect)

    mon_grn = gevent.spawn(_mon)
    gevent.sleep(0.2)

    yield

    mon_grn.kill()
    job.send(True)
    job.wait()
