import collections
import contextlib
import functools
import gevent
import inspect
import logging
import time
import traceback
import types


class SmartLoggerAdapter(logging.LoggerAdapter):
    name = None

    def __init__(self, logger, extra):
        super(SmartLoggerAdapter, self).__init__(logger, extra)
        self.name = logger.name

    def get_child(self, *args, **kwargs):
        return type(self)(self.logger.getChild(*args, **kwargs), self.extra)

    def get_parent(self):
        get_parent = getattr(self.logger, 'get_parent', None)
        parent = get_parent() if get_parent else self.logger.parent
        if parent is None:
            return None
        return type(self)(parent, self.extra)

    def get_local(self, key):
        grn = gevent.getcurrent()
        local = getattr(grn, 'slocal', getattr(grn, 'tlocal', {}))
        return local.get(key)

    getChild = get_child


class ComponentLoggerAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        prefix = self.extra.get('prefix', None)
        if not prefix:
            return msg, kwargs
        return prefix + msg, kwargs


class ComponentLoopLoggerAdapter(ComponentLoggerAdapter):
    def process(self, msg, kwargs):
        return '[%s]: %s' % (self.extra.get('loop_name', ''), msg), kwargs


class Component(object):
    __default = object()

    class WakeUp(BaseException):
        pass

    def __init__(
        self,
        parent=None,
        logname=__default,
        log_msg_prefix=__default
    ):
        self.loops = collections.OrderedDict()
        self.childs = []
        self.parent = parent

        if parent:
            parent.add_child(self)

        self._logname = (
            self.__class__.__name__
            if logname is self.__default
            else logname
        )
        self._log_msg_prefix = '' if log_msg_prefix is self.__default else log_msg_prefix
        self._make_logger()
        self._init_loops()

    def _make_logger(self):
        if not self.parent:
            base_logger = logging.getLogger('')
            base_logger.name = 'main'
            # base_logger = adapt_logger(base_logger)
        else:
            base_logger = self.parent.log

        if self._logname:
            base_logger = base_logger.getChild(self._logname)

        if self._log_msg_prefix:
            prefix = '[%s]: ' % (self._log_msg_prefix, )
        else:
            prefix = ''

        self.log = ComponentLoggerAdapter(
            base_logger,
            {'prefix': prefix}
        )

    def _init_loops(self):
        for key in dir(self):
            if key.startswith('__'):
                continue
            meth = getattr(self, key)
            if getattr(meth, 'is_a_loop', False):
                if meth not in self.loops:
                    self.loops[meth] = None

    def start(self):
        for meth, loop in self.loops.items():
            assert loop is None, '%r %r already running' % (meth, loop)
            meth.__func__.grn = self.loops[meth] = gevent.spawn(meth, **meth.extra)

        for child in self.childs:
            child.start()

        return self

    def stop(self):
        for child in self.childs:
            child.stop().join()

        for meth, loop in self.loops.items():
            if loop:
                loop.kill(gevent.GreenletExit)

        return self

    def join(self):
        if self.childs:
            joiners = []
            for child in self.childs:
                if child:
                    joiners.append(gevent.spawn(child.join))

            joiners.extend([loop for loop in self.loops.values() if loop is not None])

            gevent.joinall(joiners)
        else:
            gevent.joinall([loop for loop in self.loops.values() if loop is not None])
        return self

    def add_child(self, component):
        assert isinstance(component, Component)

        for child in self.childs:
            assert type(child) != type(component), 'Possible leak with %r and %r' % (component, self)

        self.childs.append(component)

    class TimerResult(object):
        __slots__ = 'spent',

    @contextlib.contextmanager
    def timer(self):
        ts = time.time()
        result = self.TimerResult()

        try:
            yield result
        finally:
            result.spent = time.time() - ts

    def add_loop(self, meth, **kwargs):
        loop = type(self).green_loop(meth, **kwargs)
        self.loops[loop] = None
        return loop

    @classmethod
    def green_loop(cls, meth=None, logname=None, **kwargs):
        if meth is None:
            return functools.partial(cls.green_loop, logname=logname, **kwargs)

        ism = isinstance(meth, types.MethodType)

        @functools.wraps(meth)
        def _loop(self, *args, **kwargs):
            margs, varargs, varkw, defaults = inspect.getargspec(meth)

            prefix = logname or meth.__name__
            if self._log_msg_prefix:
                prefix = '%s, %s' % (self._log_msg_prefix, prefix)

            log = ComponentLoggerAdapter(self.log, {'prefix': '[oo][%s]: ' % (prefix, )})
            log.debug('Started')

            if 'log' in margs:
                kwargs['log'] = log

            pass_wait = 'waited' in margs
            pass_wokeup = 'wokeup' in margs

            wait = 0
            wait_real = 0

            while True:
                try:
                    try:
                        if pass_wait:
                            kwargs['waited'] = wait_real

                        if pass_wokeup:
                            if ism:
                                kwargs['wokeup'] = _loop.__func__.waking
                            else:
                                kwargs['wokeup'] = _loop.waking

                        wait_real = 0

                        if ism:
                            _loop.__func__.waking = False
                            _loop.__func__.waiting = False
                            wait = meth(*args, **kwargs)
                        else:
                            _loop.waking = False
                            _loop.waiting = False
                            wait = meth(self, **kwargs)
                    except self.WakeUp:
                        log.info('Woke up (not handled by loop directly)')
                        raise

                    if ism:
                        _loop.__func__.waiting = True
                    else:
                        _loop.waiting = True

                    ts = time.time()
                    try:
                        gevent.sleep(wait or 0)
                    finally:
                        wait_real = time.time() - ts

                except gevent.GreenletExit:
                    log.info('Caught GreenletExit, exiting...')
                    return
                except self.WakeUp:
                    continue
                except BaseException:
                    log.error('Unhandled exception in %r (%r)' % (meth, logname))
                    log.error(traceback.format_exc())
                    gevent.sleep(1)  # avoid busy loops
                    wait = 0

        cls.make_loop(_loop, kwargs)

        if ism:
            _loop = types.MethodType(_loop, meth.__self__)

        return _loop

    @classmethod
    def make_loop(cls, meth, extra):
        meth.is_a_loop = True
        meth.waking = False
        meth.waiting = False
        meth.grn = None
        meth.extra = extra

        def _wake_up():
            if meth.waking:
                return
            if not meth.grn:
                return
            meth.grn.kill(cls.WakeUp, block=False)
            meth.waking = True

        meth.wakeup = _wake_up

        return meth
