from .. import monkey_patch as skbn_monkey_patch

skbn_monkey_patch()


import argparse
import errno
import faulthandler
import os
import sys
import threading
import time

import gevent
import gevent.lock
import gevent.socket

try:
    from setproctitle import setproctitle
except ImportError:
    setproctitle = None  # noqa

from .world import World
from .lookup import Lookup

from ..component import Component
from ..procrunner import ChildProc
from ...kernel_util import logging
from ...rpc import errors as rpcerrors
from ...rpc.server import RPC, Server as RPCServer

from ..diskio.io import IO
from ..diskio.cache import PathCache


class SkybitDaemon(Component):
    def __init__(
        self,
        uid, desc,
        master_cli,
        rpc_uds,
        data_uds,
        proxy_uds,
        direct_io,
    ):
        super(SkybitDaemon, self).__init__(logname='skbt')

        self.uid = uid
        self.desc = desc
        self.master_cli = master_cli

        self.rpc = None
        self.rpc_obj = None
        self.rpc_uds = rpc_uds
        self.rpc_server = None

        self.data_uds = data_uds
        self.proxy_uds = proxy_uds

        self.direct_io = direct_io

        self._log_handlers = None

        self.resource_lookup_lock = gevent.lock.Semaphore(1)
        self.resource_lookup_job = None

        self.io = IO(4, parent=self)
        self.pathcache = PathCache(master_cli, parent=self)

        self.lookup = None

    def start(self):
        self.rpc_obj = SkybitDaemonRPC(self)
        self.rpc = RPC(self.log)
        self.rpc_server = RPCServer(self.log, backlog=gevent.socket.SOMAXCONN, max_conns=1000, unix=self.rpc_uds)
        self.rpc_server.register_connection_handler(self.rpc.get_connection_handler())

        self.rpc.mount(self.rpc_obj.ping)
        self.rpc.mount(self.rpc_obj.stats)

        self.lookup = Lookup(self.lookup_resource, self.get_block_by_md5)

        self.world = World(uid=self.uid, desc=self.desc, uds=self.data_uds, uds_proxy=self.proxy_uds, parent=self)
        self.world.enable_daemon_mode()
        self.world.set_lookupper(self.lookup)

        self.world.set_memory_limit(1024 ** 3)      # do not make new hanldes after 1gb
        self.world.set_handles_clean_watchdog(600)  # check handles clean routine
        self.world.set_autoreset_nohandles(600)     # auto restart on high memory usage

        super(SkybitDaemon, self).start()
        self.rpc_server.start()

    def stop(self):
        if self.rpc_server:
            self.rpc_server.stop()

        super(SkybitDaemon, self).stop()

    def lookup_resource(self, uid):
        with self.resource_lookup_lock:
            try:
                if not self.resource_lookup_job:
                    self.resource_lookup_job = self.master_cli.call('skybit_lookup_resource')
                    assert self.resource_lookup_job.next() == 'ready'
                self.resource_lookup_job.send(((uid, ), {}))
                return self.resource_lookup_job.next()
            except rpcerrors.CallFail as ex:
                self.log.warning('Unable to lookup resource in master: %s', str(ex))
                return None
            except Exception as ex:
                self.log.error('Unable to lookup resource: %s', str(ex))
                self.resource_lookup_job = None
                return

    def get_block_by_md5(self, md5, start, length, memory_segment, sha1hash):
        data = None

        for path in self.pathcache.get_paths(md5):
            try:
                data = self.io.read_block(None, path, start, length, direct=self.direct_io)
            except (OSError, IOError) as ex:
                if ex.errno in (
                    666,
                    errno.ENOENT,
                    errno.EPERM,
                    errno.EACCES,
                ):
                    self.log.info('%s: bad file (%s)', path, str(ex))

                    try:
                        mtime = os.stat(path).st_mtime
                    except:
                        mtime = None

                    self.pathcache.bad_file(path, md5, mtime)
                else:
                    self.log.warning('%s: unexpected error (%s)', path, str(ex))

                continue
            else:
                break

        if data:
            assert memory_segment.offset == memory_segment.size == 0, 'Memory segment is already used'
            memory_segment.write(data)
            return True
        else:
            raise Exception('Unable to get data')


class SkybitDaemonRPC(object):
    def __init__(self, skybit):
        self.skybit = skybit

    def ping(self):
        assert self.skybit.world.check(), 'world.check() failed'
        return 'OK'

    def stats(self):
        if self.skybit.lookup and self.skybit.lookup.shared_memory:
            mem = self.skybit.lookup.shared_memory
            return {'memory': {'size': mem.size, 'free': mem.free_queue.qsize() * 4 * 1024 * 1024}}
        else:
            return {'memory': {'size': 0, 'free': 0}}


def initialize_logging():
    logging.renameLevels(1)

    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.DEBUG)

    if sys.stdout.isatty():
        formatter = logging.ColoredFormatter(
            fmt='%(asctime)s %(levelname)s %(name)s  %(message)s',
            fmtName='[%(name)-26s]',
            fmtLevelname='[%(levelname)-1s]',
            fmtAsctime='%(datetime)s.%(msecs)003d',
            hungryLevels={
                'info': ['message'],
                'debug': ['message'],
                'warning': ['message'],
                'error': ['message', 'name']
            }
        )
        console_handler.setFormatter(formatter)
    else:
        formatter = logging.Formatter(
            fmt='%(levelname)s  %(name)-20s  %(message)s'
        )
        console_handler.setFormatter(formatter)

    root_logger = logging.getLogger('')
    root_logger.setLevel(logging.DEBUG)
    root_logger.addHandler(console_handler)


def _child_check_parent():
    while True:
        if os.getppid() == 1:
            os._exit(1)
        time.sleep(1)


def main():
    faulthandler.enable()
    if setproctitle is not None:
        setproctitle('skybone-skybit')

    parser = argparse.ArgumentParser()
    parser.add_argument('--master-uds', required=True)

    parser.add_argument('--rpc-uds', required=True)
    parser.add_argument('--rpc-uds-abstract', action='store_true')
    parser.add_argument('--child', action='store_true')

    parser.add_argument('--data-uds', required=True)
    parser.add_argument('--data-uds-abstract', action='store_true')
    parser.add_argument('--proxy-uds', required=True)
    parser.add_argument('--proxy-uds-abstract', action='store_true')

    parser.add_argument('--world-uid', required=True)
    parser.add_argument('--world-desc', required=True)

    parser.add_argument('--direct-io', action='store_true')

    args = parser.parse_args()

    if args.child:
        thr = threading.Thread(target=_child_check_parent)
        thr.daemon = True
        thr.start()

    initialize_logging()

    cproc = ChildProc(name='skybit', master_uds=args.master_uds)
    cproc.start()

    sd = SkybitDaemon(
        uid=args.world_uid,
        desc=args.world_desc,
        master_cli=cproc.master_cli,
        rpc_uds=('\0' if args.rpc_uds_abstract else '') + args.rpc_uds,
        data_uds=('\0' if args.data_uds_abstract else '') + args.data_uds,
        proxy_uds=('\0' if args.proxy_uds_abstract else '') + args.proxy_uds,
        direct_io=args.direct_io,
    )

    try:
        sd.start()

        while True:
            gevent.sleep(3600)
    except KeyboardInterrupt:
        sys.stderr.write('Interrupted')
        sd.stop()
        sd.join()
        cproc.stop()
        cproc.join()


if __name__ == '__main__':
    main()
