import contextlib

import gevent
import gevent.lock


class InterruptsProxy(object):
    def __init__(self, parent, current):
        self.parent = parent
        self.current = current

    @contextlib.contextmanager
    def suspend(self):
        self.parent.suspended[self.current] = None
        try:
            yield
        finally:
            ex = self.parent.suspended.pop(self.current)
            if ex:
                raise ex


class Interrupts(object):
    def __init__(self):
        self.waiters = {}
        self.interrupts = {}
        self.suspended = {}
        self.lock = gevent.lock.Semaphore(1)

    @contextlib.contextmanager
    def interruptable(self, events):
        current = gevent.getcurrent()

        proxy = InterruptsProxy(self, current)

        try:
            locked = False
            with self.lock:
                for event in events:
                    waiters = self.waiters.setdefault(event, set())
                    waiters.add(current)

            yield proxy

            if not self.lock.locked():
                # Dont allow to interrupt us anymore
                self.lock.acquire()  # this can be interrupted as well
                locked = True

            ex = None

        except BaseException as ex:
            raise ex from None

        finally:
            if not self.lock.locked():
                self.lock.acquire()
                locked = True

            reraise = ex

            try:
                if current in self.interrupts:
                    # We have pending interruptions for this greenlet
                    interrupt, ev = self.interrupts.pop(current)

                    while ex != interrupt:
                        # Wait for interrupt to happen
                        try:
                            gevent.sleep(1000)
                        except BaseException:
                            continue

                else:
                    ev = None

                if ev:
                    ev.set()

                # Interrupt happend, signaling event
                for event in events:
                    self.waiters[event].discard(current)

            finally:
                if locked:
                    self.lock.release()

                if reraise:
                    raise reraise

    def interrupt(self, event):
        with self.lock:
            if event not in self.waiters:
                return

            wait_evs = []
            for waiter in list(self.waiters[event]):
                ev = gevent.event.Event()
                interrupt = event('killing %r' % (waiter, ))

                assert waiter not in self.interrupts

                if waiter in self.suspended:
                    self.suspended[waiter] = interrupt
                else:
                    self.interrupts[waiter] = (interrupt, ev)
                    wait_evs.append(ev)
                    waiter.kill(interrupt, block=0)

            reraise = None
            for ev in wait_evs:
                while True:
                    try:
                        ev.wait()
                    except BaseException as ex:
                        if not reraise:
                            reraise = ex  # catch only first error
                        continue
                    else:
                        break

            if reraise:
                raise reraise
