from __future__ import division

import gevent
import time


class Bucket(object):
    """
    Leaking bucket with specified rate in B/s and max allowed burst.

    Burst is the minimal refill speed.

    In 8192 bursts:

        - leak 100 bytes (left 100)
        - leak 100 bytes (left 0)
        - wait burst / rate
        - refill 8192
        - leak 100 bytes (left 8092)
        - etc.

    So, it will look like we send 8192 bytes, then sleep 8192/rate seconds, then
    again send 8192 bytes, etc.

    This is done for 2 reasons:
        - much less function calls in python code (25-30% less cpu usage)
        - we will never send too small packets over network ;)
    """

    def __init__(self, rate, burst=8192, log=None):
        self.rate = rate
        self.burst = burst
        self.capacity = rate
        self.last_leak = 0
        self._lock = gevent.lock.Semaphore(1)

    def _leak(self, amount):
        now = time.time()
        elapsed = now - self.last_leak

        self.capacity = min(self.rate, self.capacity + int(self.rate * elapsed))
        self.last_leak = now

        allowed = min(amount, self.capacity)

        self.capacity -= allowed

        return allowed

    def leak(self, amount):
        with self._lock:
            allowed = 0
            need_more = amount

            allowed += self._leak(need_more)
            need_more = amount - allowed

            if need_more > 0:
                need_wait = max(self.burst, need_more) / self.rate
                gevent.sleep(need_wait)

                allowed += self._leak(need_more)

            return allowed
