import bisect
import gevent
import random
import time

from collections import OrderedDict

from .peer import Peer, PeerAddress

from .component import Component


class PeerPriorityQueue(gevent.queue.Queue):
    def _init(self, maxsize):
        self.queue = []

    def _put(self, item):
        assert len(item) == 2

        pos = bisect.bisect(self.queue, (-item[0], ))

        if pos == len(self.queue):
            # Should add last
            wqueue = []
            self.queue.append((-item[0], wqueue))
        elif self.queue[pos][0] == -item[0]:
            wqueue = self.queue[pos][1]
        else:
            wqueue = []
            self.queue.insert(pos, (-item[0], wqueue))

        wqueue.append(item[1])

    def _get(self):
        if len(self.queue) == 0:
            raise IndexError()

        weight, wqueue = self.queue[0]
        item = (-weight, wqueue.pop(0))

        if len(wqueue) == 0:
            self.queue.pop(0)

        return item

    def get(self, *args, **kwargs):
        return super(PeerPriorityQueue, self).get(*args, **kwargs)[1]

    def put(self, weight, peer):
        item = (weight, peer)
        return super(PeerPriorityQueue, self).put(item)

    def discard(self, weight, peer):
        if len(self.queue) == 0:
            return False

        pos = bisect.bisect(self.queue, (weight, []))

        if len(self.queue) == pos:
            return False

        wweight, wqueue = self.queue[pos]
        if wweight == weight:
            # Weight queue already exists
            try:
                wqueue.remove(peer)
            except ValueError:
                return False
            else:
                if len(wqueue) == 0:
                    del self.queue[pos]
                return True

        return False


class PeerDeferredList(list):
    def put(self, ts, peer):
        bisect.insort(self, (ts, peer))

    def discard(self, ts, peer):
        item = (ts, peer)
        pos = bisect.bisect(self, item)
        prevpos = pos - 1

        if prevpos >= 0 and self[prevpos] == item:
            del self[prevpos]
            return True

        return False

    def get_minimal_time(self):
        if self:
            return self[0][0]

    def get_minimal_time_from_now(self):
        minimal_time = self.get_minimal_time()
        if minimal_time is not None:
            return minimal_time - time.time()

    def pop(self, pos=None):
        return super(PeerDeferredList, self).pop(pos)[1]


class PeerConnectCandidates(Component):
    def __init__(self, parent=None):
        self.queue = PeerPriorityQueue()
        self.deferred = PeerDeferredList()

        self._map = {}  # peer => (deferred_time, queued_weight)

        self.cnt_deferred = 0
        self.cnt_queued = 0

        super(PeerConnectCandidates, self).__init__(logname='nxt', parent=parent)

    def queued(self, peer):
        return peer in self._map

    def discard(self, peer):
        old_deferred, old_weight = self._map.pop(peer, (None, None))

        if old_deferred is not None:
            assert self.deferred.discard(old_deferred, peer)
            self.cnt_deferred -= 1

        if old_weight is not None:
            assert self.queue.discard(old_weight, peer)
            self.cnt_queued -= 1

    def put(self, peer, defer=None):
        if isinstance(peer, Peer):
            if peer.conn and defer is None:
                # Probably peer already connected to us by itself
                self.put(peer, 10)
                return

        self.discard(peer)

        if defer is not None:
            defer_time = time.time() + defer
        else:
            defer_time = None

        if defer_time is not None:
            self._map[peer] = (defer_time, None)
            self.deferred.put(defer_time, peer)
            self.cnt_deferred += 1
            self.deferer.wakeup()
        else:
            self._map[peer] = (None, peer.weight)
            self.queue.put(peer.weight, peer)
            self.cnt_queued += 1

        self.log_candidates('put')

    def get(self, block=True, timeout=None):
        peer = self.queue.get(block=block, timeout=timeout)
        self.cnt_queued -= 1

        self._map.pop(peer)

        self.log_candidates('get')
        return peer

    @Component.green_loop
    def deferer(self, log):
        sleeptime = self.deferred.get_minimal_time_from_now()

        if sleeptime is None:
            return 60

        elif sleeptime > 0:
            return sleeptime

        peer = self.deferred.pop(0)
        self.cnt_deferred -= 1

        self._map.pop(peer, None)
        self.put(peer)

    def log_candidates(self, reason):
        self.log.info('cst: [%s] nxt: %d  deferred: %d', reason, self.cnt_queued, self.cnt_deferred)


class PeerCollection(Component):
    def __init__(self, parent=None):
        self.map = {}               # map uid => peer
        self.ipmap = {}             # ip, port => peer
        self.addrs = {}             # map ip, port => address
        self.addrs_next = set()     # address which was not tried yet

        # "keep going" states
        self.connecting = set()     # tcp conn made, but no activation/handshake yet
        self.connected = set()      # handshake completed & activated

        # "retry sometime" states
        self.disconnected = set()   # disconnected after successful connection
        self.no_slot = set()        # no slots

        # "do not connect" states
        # This one is essentially same as "disconnected" but without any outgoing
        # connections made anymore, but we allow incoming connections from that peer (and "no want" state
        # could be reset that way)
        self.no_want = set()        # we dont want anything from this peer right now

        # Final states
        self.no_resource = set()    # no such resource here

        # This one is not kinda used atm. It is meant to be used if we are downloading from seeder and
        # become seeder ourselves without closing all connections (thus, we are keep seeding)
        # So we would disconnect seeders in that case. But it is not gonna happen as of now.
        self.no_need = set()        # bidirectional state: neither we neither that peer should connect to us again

        self.collections = OrderedDict((
            ('connecting', self.connecting),
            ('connected', self.connected),
            ('disconnected', self.disconnected),
            ('no_resource', self.no_resource),
            ('no_slot', self.no_slot),
            ('no_want', self.no_want),
            ('no_need', self.no_need),
        ))

        super(PeerCollection, self).__init__(logname='prs', parent=parent)

        self.nxt = PeerConnectCandidates(parent=self)

    def add(self, peer):
        """ Add new peer to collection. Used to add new peers"""

        self.map[peer.uid] = peer

        if peer.ou_address:
            # If this one is outgoing peer -- add it's outgoing address
            self.ipmap[peer.ou_address] = peer

    def add_address(self, addr, defer=None):
        """
        Add plain address, which usually comes from tracker.

        Address is added in two cases:
            - new address from tracker
            - we were not able to connect/handshake peer, so we push connect address back

        This is *NOT* called for incoming peers, so ipmap will contain only ips/ports we
        got from tracker
        """

        assert isinstance(addr, PeerAddress)

        if addr in self.ipmap:
            item = self.ipmap[addr]
        elif (addr.ip, addr.port) in self.addrs:
            item = self.addrs[addr.ip, addr.port]
        else:
            self.addrs[addr.ip, addr.port] = addr
            item = addr

        self.addrs_next.add(addr)

        # Queue connect only if not queued already.
        if not self.nxt.queued(item):
            self.log.debug('add_address: new item %r, queue defer=%s', item, defer)
            self.nxt.put(item, defer)
        else:
            self.log.debug('add_address: new item %r, already queued', item)

    def get(self, uid):
        return self.map.get(uid)

    def get_next(self, timeout=None):
        try:
            return self.nxt.get(timeout=timeout)
        except gevent.queue.Empty:
            return None

    def collect(self, peer):
        if peer.state == Peer.CONNECTING:
            # This collection called after tcp connection is made, but before
            # handshake and connection activation. Called once before activation
            target = self.connecting
            retry_in = False

        elif peer.state == Peer.CONNECTED:
            # This is called once after connection has been successfully activated
            # And used to populate peer into self.connected collection
            target = self.connected
            retry_in = False

        elif peer.state == Peer.DISCONNECTED:
            target = self.disconnected
            if peer.ou_address:
                retry_in = 3
            else:
                retry_in = False

        elif peer.state == Peer.NO_SKYBIT:
            # Connected to peer, but no "skybit" capability sent in handshake
            # Never try to connect them again
            target = self.no_want
            retry_in = False

        elif peer.state == Peer.NO_RESOURCE:
            # Peer reports he has no resource by specified uid. Right now we will
            # never retry.
            target = self.no_resource
            if peer.ou_address:
                retry_in = 60
            else:
                retry_in = False

        elif peer.state == Peer.NO_SLOT:
            # We connected to peer, but it reports where is no free tcp slots.
            # Be calm, disconnect and retry later in 30..90 seconds
            target = self.no_slot

            if peer.ou_address:
                retry_in = 30 + random.randint(0, 60)
            else:
                retry_in = False

        elif peer.state == Peer.NO_NEED:
            # Upload to upload connection, usually happens then we finish download.
            target = self.no_need
            retry_in = False

        elif peer.state == Peer.NO_WANT:
            # We dont want anything from this peer yet
            target = self.no_want
            retry_in = False

        else:
            self.log.critical('Unable to collect peer %r with state: %r', peer, peer.state)
            return

        if target is not None:
            for collection in self.collections.values():
                collection.discard(peer)
            target.add(peer)

        if retry_in is not False:
            self.log.debug('Defer peer %r to nxt in %r secs', peer, retry_in)
            self.nxt.put(peer, defer=retry_in or None)

        self.log_colstats('change')  # 6-char reason length is important here!

    def _short_collection_name(self, name):
        return {
            'connecting': 'preact',
            'connected': 'active',
            'disconnected': 'discnctd',
            'no_resource': 'nores',
            'no_slot': 'noslt',
            'no_want': 'nownt',
            'no_need': 'noneed',
        }.get(name, name)

    def log_colstats(self, reason):
        collection_stats = OrderedDict([(name, len(collection)) for name, collection in self.collections.items()])
        self.log.debug(
            'cst: addrs:%d next:%d  %s  %s',
            len(self.addrs), len(self.addrs_next),
            ' '.join(
                '%s:%d' % (self._short_collection_name(name), stat)
                for name, stat
                in collection_stats.items()
            ),
            reason
        )
