from __future__ import division, print_function

import math
import random
import hashlib

import gevent

from .component import Component


class Piece(object):
    __slots__ = (
        'length', 'sha1hash'
    )

    def __init__(self, length, sha1hash):
        self.length = length
        self.sha1hash = sha1hash


class Item(object):
    __slots__ = (
        'name', 'offset',
    )


class Link(object):
    __slots__ = Item.__slots__ + (
        'target', 'symbolic'
    )


class Data(Item):
    __slots__ = Item.__slots__ + (
        'executable', 'md5hash', 'size', 'pieces'
    )

    def check(self, idx, memory_segment):
        checksum = self.pieces[idx].sha1hash
        tsum = hashlib.sha1()
        tsum.update(memory_segment.read())
        memory_segment.rewind()
        return tsum.hexdigest() == checksum


class PieceArray(object):
    __slots__ = (
        'pieces',
    )

    def __init__(self, pieces):
        self.pieces = pieces

    def __repr__(self):
        length = len(self.pieces)
        done = self.pieces.count(True)

        return '<PieceArray: %d pieces, %d%% done>' % (len(self.pieces), (done / length) * 100)

    def set_done(self, idx):
        assert not self.pieces[idx], 'Piece %d already marked as done!' % (idx, )
        self.pieces[idx] = True


class PieceMap(Component):
    def __init__(self, parent=None):
        super(PieceMap, self).__init__(logname='pmap', parent=parent)

        self.data = {}           # md5  => (idx, data)  # DATA IDX IS DATA NUM, NOT PIECE INDEX
        self.data_by_idx = []    # idx  => data (list)
        self.path_by_data = {}   # data => [paths]
        self.piece_array = None  # idx  => (True|False)

        self.loaded = gevent.event.Event()

    def load_head(self, head):
        # Load all data, symlinks and file mirrors
        piece_array = []

        idx_offset = 0

        for name, props in sorted(head['structure'].items()):
            if props['resource']['type'] != 'torrent':
                continue

            data_md5 = props['md5sum'].hex()

            if data_md5 not in self.data:
                data_sha1hash = props['resource']['id']

                data = Data()
                data.name = name
                data.executable = bool(props['executable'])
                data.md5hash = data_md5
                data.size = props['size']
                data.pieces = []
                data.offset = len(self.data_by_idx)

                torrent_info = head['torrents'][data_sha1hash]['info']
                piece_hashsums = torrent_info['pieces'].hex()
                piece_length = torrent_info['piece length']

                for idx in range(
                    int(math.ceil(data.size / piece_length))
                ):
                    sha1hash = piece_hashsums[idx * 40:(idx + 1) * 40]
                    data.pieces.append(Piece(piece_length, sha1hash))
                    piece_array.append(False)
                    self.data_by_idx.append(data)

                self.data[data_md5] = (idx_offset, data)
                idx_offset += len(data.pieces)
            else:
                data = self.data[data_md5][1]

            self.path_by_data.setdefault(data, []).append(name)

        self.piece_array = PieceArray(piece_array)
        self.loaded.set()

    def piece_done(self, piece_idx):
        self.piece_array.set_done(piece_idx)

    def all_done(self):
        for idx in range(len(self.piece_array.pieces)):
            self.piece_done(idx)

    def get_map(self):
        return self.piece_array


class PiecePicker(object):
    def __init__(self):
        self.pmap = None
        self.requests = {}  # key => set() of idx'es

        self.pieces_needed = set()       # each piece we need to download goes here
        self.pieces_requested = set()    # once we request some piece
        self.pieces_checked = set()      # once we receive and check piece (but do not write yet)

    def load_pmap(self, pmap, partial=None):
        self.pmap = pmap

        if partial:
            partial_pieces = set()

            for idx, data in enumerate(pmap.data_by_idx):
                needed = False
                for path in pmap.path_by_data[data]:
                    if path in partial:
                        needed = True
                        break

                if needed:
                    for i in range(idx, idx + len(data.pieces)):
                        partial_pieces.add(i)

        for idx, piece_done in enumerate(self.pmap.piece_array.pieces):
            if not piece_done:
                if partial:
                    if idx in partial_pieces:
                        self.pieces_needed.add(idx)
                else:
                    self.pieces_needed.add(idx)

    def set_peer_pieces(self, key, pieces):
        assert key not in self.requests
        info = [set(), set(), set()]  # HAS, WE_WANT, REQUESTED

        for idx, piece_done in enumerate(pieces):
            if piece_done:
                info[0].add(idx)
                if idx in self.pieces_needed:
                    info[1].add(idx)

        self.requests[key] = info

    def peer_have(self, key, idx):
        info = self.requests[key]
        info[0].add(idx)

        if idx in self.pieces_needed:
            info[1].add(idx)

    def get_piece_for_request(self, key):
        info = self.requests[key]

        key_req = info[2]

        if len(key_req) >= 2:
            # We already have 2 pieces requested, should not ask for any other
            return

        if not info[1]:
            # We dont want any pieces
            return

        idx = random.choice(list(info[1]))
        key_req.add(idx)
        info[1].remove(idx)

        self.pieces_needed.remove(idx)
        self.pieces_requested.add(idx)

        # Remove from other peers WE_WANT, so we will not double request this
        for okey, oinfo in self.requests.items():
            if okey == key:
                continue
            oinfo[1].discard(idx)

        return idx

    def discard_requests(self, key):
        discarded = []

        if key in self.requests:
            for idx in self.requests.pop(key)[2]:
                # Add to other conns WE_WANT
                for okey, oinfo in self.requests.items():
                    if idx in oinfo[0]:    # If peer Has it
                        oinfo[1].add(idx)  # set we want it again

                self.pieces_needed.add(idx)
                self.pieces_requested.remove(idx)
                discarded.append(idx)

        return discarded

    def piece_checked(self, key, idx):
        peer_has, we_want, requested = self.requests[key]

        assert idx in requested
        assert idx not in we_want

        requested.remove(idx)
        assert idx not in self.pieces_needed
        assert idx not in self.pieces_checked

        self.pieces_requested.remove(idx)
        self.pieces_checked.add(idx)

    def piece_unchecked(self, key, idx):
        # If we checked piece, but failed during writing

        if key in self.requests:
            peer_has, we_want, requested = self.requests[key]
            assert idx not in requested
            assert idx not in we_want

            we_want.add(idx)

        assert idx not in self.pieces_needed
        assert idx not in self.pieces_requested
        assert idx in self.pieces_checked

        self.pieces_checked.remove(idx)
        self.pieces_needed.add(idx)

    def piece_done(self, key, idx):
        if key is None:
            self.pieces_needed.remove(idx)

            # We may already copied needed pieces to some key's requests
            # So, drop it now. Note that oinfo[1] will not have this piece
            # if other peer has not it (after sending piecemap to us).
            # Whats why we are using discard() instead of remove()
            for okey, oinfo in self.requests.items():
                oinfo[1].discard(idx)

            return

        if key in self.requests:
            # Key may be not in requests if peer disconnected before we wrote piece
            peer_has, we_want, requested = self.requests[key]

            assert idx not in requested
            assert idx not in we_want

        assert idx not in self.pieces_needed
        assert idx not in self.pieces_requested
        assert idx in self.pieces_checked

        self.pieces_checked.remove(idx)

        self.pmap.piece_done(idx)
