import gevent.queue
import gevent.socket
import os
import psutil
import struct
import sys
import time
import traceback

from .logger import WorldLoggerAdapter
from .handle import Handle, NoResourceHandle, InfoHashDetectHandle
from .netbucket import Bucket

from .component import Component


class World(Component):
    def __init__(
        self, uid, desc, port=0, uds=None, uds_proxy=False, extended_logging=False,
        limit_dl=None, limit_ul=None,
        parent=None
    ):
        super(World, self).__init__(parent=parent, logname='wrld')

        self.uid = uid
        self.desc = desc

        self.handles = {}
        self._handles_clean_ts = time.time()
        self._last_handle_ts = time.time()

        self.handle_no_resource = None

        self.sock = None
        self.port = port
        self.uds = uds
        self.uds_proxy = uds_proxy

        if self.uds is not None:
            assert port == 0

        if extended_logging:
            self.log = WorldLoggerAdapter(self.log, {'world_uid': self.uid, 'world_desc': self.desc})

        self.lookupper = None
        self.memory_limit = None
        self.handles_clean_watchdog = None
        self.autoreset_nohandles = None

        self.loops.pop(self.clean_handles)

        self.bucket_in = None
        self.bucket_ou = None

        if limit_dl:
            self.bucket_in = Bucket(limit_dl)

        if limit_ul:
            self.bucket_ou = Bucket(limit_ul)

    def set_lookupper(self, lookupper):
        self.lookupper = lookupper

    def set_memory_limit(self, size):
        self.memory_limit = size

    def set_handles_clean_watchdog(self, time):
        self.handles_clean_watchdog = time

    def set_autoreset_nohandles(self, time):
        self.autoreset_nohandles = time

    def enable_daemon_mode(self):
        self.loops[self.clean_handles] = None

    def start(self):
        self.sock = self.make_socket(uds=self.uds)

        if not self.uds:
            self.log.debug('Bind main socket on %d', self.port)
            self.sock.bind(('', self.port))
        else:
            self.log.debug('Bind main socket at %r', self.uds)
            if not self.uds.startswith('\0'):
                try:
                    os.unlink(self.uds)
                except gevent.GreenletExit:
                    raise
                except Exception:
                    pass

            self.sock.bind(self.uds)

            if not self.uds.startswith('\0'):
                os.chmod(self.uds, 0o666)

        self.handle_no_resource = NoResourceHandle(self, self.log.getChild('hdl.nores'))
        self.handle_infohash_detect = InfoHashDetectHandle(self, self.log.getChild('hdl.detect'))

        self.sock.listen(gevent.socket.SOMAXCONN)

        return super(World, self).start()

    def add_child(self, component):
        raise NotImplementedError('Never add world childs!')

    def check(self):
        if self.uds and sys.platform != 'cygwin':
            sk = gevent.socket.socket(gevent.socket.AF_UNIX, gevent.socket.SOCK_STREAM)
            with gevent.Timeout(10) as tout:
                try:
                    sk.connect(self.uds)
                    sk.sendall('PING' + ' ' * 35)
                    data = sk.recv(4)
                    sk.close()
                    assert data == 'PONG', 'Got invalid data from socket %r' % (data, )
                except gevent.Timeout as ex:
                    if ex != tout:
                        raise

                    self.log.critical('uds sock timed out')
                    self.log.critical(traceback.format_exc())
                    return False

        return True

    def handle(self, uid, priority=None, on_stop=None):
        if uid not in self.handles:
            handle = Handle(self, uid, log=self.log.getChild('hdl'), on_stop=on_stop, net_priority=priority)
            self.handles[uid] = handle
            self._last_handle_ts = time.time()
        return self.handles[uid]

    @Component.green_loop
    def clean_handles(self, log):
        closed = 0
        drop = set()
        for uid, handle in self.handles.items():
            if handle.idle_time() > 60:
                drop.add(uid)

        for uid in drop:
            handle = self.handles[uid]
            log.debug(
                'Closing handle %s: idle for %d secs (left %d)',
                uid[:8], handle.idle_time(), len(self.handles) - 1
            )
            closed += 1
            try:
                self.handles.pop(uid).stop()
            except Exception as ex:
                log.warning('  error while closing handle: %s', ex)

        if closed > 0 or len(self.handles) > 0:
            log.debug('Left %d alive handles (closed %d)' % (len(self.handles), closed))

        self._handles_clean_ts = time.time()

        return 30

    def _handle_stopped(self, handle):
        self.handles.pop(handle.uid, None)
        try:
            handle.stop()
        except gevent.GreenletExit:
            raise
        except Exception:
            pass

    def _memory_usage(self):
        return psutil.Process(os.getpid()).get_memory_info().rss

    @Component.green_loop
    def _acceptor(self, log):
        while True:
            sock, _ = self.sock.accept()

            try:
                if self.uds:
                    with gevent.Timeout(0.2) as tout:
                        try:
                            payload = sock.recv(39)
                            if len(payload) < 39:
                                sock.close()
                                continue

                            if payload[:18] == 'SKYBONE_NETPROXY_1':
                                # Grab extra 18 bytes
                                # Trust only connections with SKYBONE_NETPROXY_1 magic ;)
                                payload = payload[18:] + sock.recv(18)
                                if len(payload) < 39:
                                    sock.close()
                                    continue

                        except gevent.Timeout as ex:
                            if ex != tout:
                                raise
                            sock.close()
                            continue

                    if payload[:4] == 'PING':
                        sock.sendall('PONG')
                        sock.close()
                        continue

                    infohash_raw, af, ip_packed, port = struct.unpack('!20sB16sH', payload)
                    infohash = infohash_raw.hex()

                    if af not in (gevent.socket.AF_INET, gevent.socket.AF_INET6):
                        log.error('Invalid address family %r', af)
                        sock.close()
                        continue

                    if af == gevent.socket.AF_INET:
                        ip_packed = ip_packed[:4]

                    ip = gevent.socket.inet_ntop(af, ip_packed)
                else:
                    ip, port = _[:2]
                    infohash = None

                payload = None

                if not infohash:
                    try:
                        infohash, payload = self.handle_infohash_detect.handle_incoming_connection(
                            sock, payload, ip, port
                        )
                    except Exception as ex:
                        log.exception(
                            '[resid:%s] Unable to handle incoming connection (detecting): %s: %s',
                            '?' * 8, type(ex).__name__, ex
                        )
                        sock.close()
                        continue

                    if infohash is None:
                        # Invalid proto, skycore pings, etc.
                        sock.close()
                        continue

                if infohash in self.handles:
                    handle = self.handles[infohash]

                elif self.lookupper:
                    if (
                        self.handles_clean_watchdog
                        and time.time() - self._handles_clean_ts > self.handles_clean_watchdog
                    ):
                        log.critical(
                            'Last handles clean ts more than %d mins in the past! Emergency exit',
                            self.handles_clean_watchdog // 60
                        )
                        os._exit(1)

                    if self.memory_limit and self._memory_usage() > self.memory_limit:
                        sock.close()
                        log.warning(
                            'Too much memory usage (%dMb, %d handles), ignoring new request',
                            self._memory_usage() / (1024 * 1024), len(self.handles)
                        )

                        if (
                            self.autoreset_nohandles
                            and time.time() - self._last_handle_ts > self.autoreset_nohandles
                        ):
                            # We have huge memory usage, but last handle created was 10 mins ago
                            # That means we have some heavy memory usage, only way to release it
                            # restart
                            log.warning(
                                'Last handle was %d mins ago -- reset to clear memory',
                                (time.time() - self._last_handle_ts) // self.autoreset_nohandles
                            )
                            os._exit(1)

                        continue

                    head, head_extra = self.lookupper.look_head(infohash)
                    if head:
                        handle = self.handle(infohash, on_stop=self._handle_stopped)
                        try:
                            handle.set_head(head, head_extra)
                            if hasattr(self.lookupper, 'get_block'):
                                handle.set_shared_memory(self.lookupper.shared_memory)
                                handle.seed(self.lookupper.get_block)
                            else:
                                handle.seed(None)

                            if hasattr(self.lookupper, 'get_link'):
                                handle.enable_dfs_mode(self.lookupper.get_link)

                        except Exception as ex:
                            log.exception('[resid:%s] Unable to create seeding handle: %s', infohash[:8], ex)
                            try:
                                self.handles.pop(infohash).stop()
                            except gevent.GreenletExit:
                                raise
                            except Exception:
                                pass

                            handle = self.handle_no_resource
                    else:
                        handle = self.handle_no_resource
                else:
                    handle = self.handle_no_resource

                try:
                    handle.handle_incoming_connection(sock, payload, ip, port)
                except Exception as ex:
                    log.exception(
                        '[resid:%s] Unable to handle incoming connection: %s: %s',
                        infohash[:8], type(ex).__name__, ex
                    )
                    sock.close()
                    continue

            except Exception as ex:
                # fyi: this is really bad -- we will sleep 1 sec to avoid busy loops
                # thus, only really CRITICAL exceptions should be here
                try:
                    sock.close()
                except Exception:
                    pass
                raise ex

    def make_socket(self, uds=False):
        """ This one used to greate both listening connection and outgoing connections. """

        if not uds:
            sock = gevent.socket.socket(gevent.socket.AF_INET6, gevent.socket.SOCK_STREAM)
            sock.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEADDR, 1)
            sock.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEPORT, 1)
            sock.setsockopt(gevent.socket.IPPROTO_IPV6, gevent.socket.IPV6_V6ONLY, 0)
            sock.setsockopt(gevent.socket.IPPROTO_TCP, gevent.socket.TCP_NODELAY, 1)
        else:
            sock = gevent.socket.socket(gevent.socket.AF_UNIX, gevent.socket.SOCK_STREAM)

        return sock

    def connect_socket(self, sock, uid, ip, port, net_priority):
        if self.uds_proxy:
            af = gevent.socket.AF_INET6 if ':' in ip else gevent.socket.AF_INET
            sock.connect(self.uds_proxy)

            sock.sendall(struct.pack(
                '!20sB16sHB',
                uid.decode('hex'), af,
                gevent.socket.inet_pton(af, ip),
                port,
                net_priority or 0
            ))

            resp = sock.recv(5)
            if len(resp) < 5:
                raise gevent.socket.error(0, 'Invalid response len from netproxy: %r' % (resp, ))

            connected, errno, errstr_len = struct.unpack('!BHH', resp)
            if not connected:
                errstr = ''
                while errstr_len:
                    data = sock.recv(errstr_len)
                    if not data:
                        raise gevent.socket.error(0, 'Invalid response message from netproxy (zero bytes)')

                    errstr_len -= len(data)
                    errstr += data

                raise gevent.socket.error(errno, errstr)
        else:
            if '.' in ip:
                ip = '::ffff:' + ip
            if net_priority:
                sock.setsockopt(gevent.socket.IPPROTO_IP, gevent.socket.IP_TOS, net_priority)
            sock.connect((ip, port, 0))
