from __future__ import absolute_import, print_function, division

import daemon
import faulthandler
import pprint
import signal
import sys
import cython

try:
    from setproctitle import setproctitle
except ImportError:
    setproctitle = None  # noqa

from . import logger


def magicPrint(*args, **kwargs):
    sep = kwargs.pop('sep', ' ')
    end = kwargs.pop('end', '\n')
    stream = kwargs.pop('file', sys.stdout)

    args = list(args)
    for idx, arg in enumerate(args):
        if not isinstance(arg, basestring):
            args[idx] = pprint.pformat(arg)

    return stream.write(sep.join(args) + end)


class Context(object):
    def __init__(self):
        # Initialize all variables
        self.log = None

        self._dctx = None
        self._progname = None

        self._initialized = False
        self._opened = False

        super(Context, self).__init__()

    def _setupPrintHack(self):
        # Hack print()
        if __file__.endswith('.so') or cython.compiled:
            return
        if 'oprint' not in __builtins__:
            __builtins__['oprint'] = __builtins__['print']
            __builtins__['print'] = magicPrint

    def initialize(
        self, progname=None, background=False, close_stdio=False, print_hack=False,
        logging=False, logging_verbosity=2, logpath=None
    ):
        assert not self._initialized, 'Context already initialized'

        self._progname = progname

        if print_hack:
            self._setupPrintHack()

        # Initialize logging subsys
        preserved_files = []
        if logging:
            preserved_files = [self.initializeLogging(logging_verbosity, logpath)]

        self._dctx = daemon.DaemonContext(  # pylint: disable=E1101
            working_directory='/',
            umask=022,
            stdout=None if close_stdio else sys.stdout,
            stderr=None if close_stdio else sys.stderr,
            stdin=None if close_stdio else sys.stdin,
            prevent_core=False,
            detach_process=background,
            signal_map=self.initializeSignalHandlers(),
            files_preserve=preserved_files,
        )

        # Disable set_signal_handlers meth, coz we have our own
        daemon.daemon.set_signal_handlers = lambda x: None

        self.__isOpen = False
        self._initialized = True

        return self

    def setSignalHandlers(self, signalHandlerMap):
        import gevent
        for signum, handler in signalHandlerMap.items():
            if handler == 1:
                handler = lambda signum: None
            try:
                gevent.signal_handler(signum, handler, signum)
            except AttributeError:
                gevent.signal(signum, handler, signum)

    def setFaultHandler(self):
        faulthandler.enable()

    def initializeSignalHandlers(self):
        signalMap = {}

        for signame in 'SIGTTIN', 'SIGTTOU', 'SIGPIPE':
            if hasattr(signal, signame):
                signalMap[getattr(signal, signame)] = None

        for signame in 'SIGTERM', 'SIGALRM', 'SIGHUP', 'SIGTSTP':
            if hasattr(signal, signame):
                signalMap[getattr(signal, signame)] = self.terminate

        return signalMap

    def initializeLogging(self, verbosity, logpath):
        _, preserved_file = logger.setup_logging(progname=self._progname, console_verbosity=verbosity, logpath=logpath)
        self.log = logger.logging.getLogger()         # Make root logger
        return preserved_file

    def terminate(self, signum):
        signame = None
        try:
            for key, value in signal.__dict__.items():
                if value == signum:
                    signame = key
                    break
            else:
                assert 0, 'not found'
        except Exception as ex:
            signame = 'unknown: {0}'.format(ex)

        raise SystemExit('Terminating on signal {0} ({1})'.format(signum, signame))

    def __enter__(self):
        assert not self._opened
        self._dctx.open()
        if self._progname and setproctitle is not None:
            setproctitle(self._progname)
        self.setSignalHandlers(self._dctx._make_signal_handler_map())
        self.setFaultHandler()
        self._opened = True
        return self

    def __exit__(self, excType, excValue, traceback):
        assert self._opened
        self._dctx.close()
        self._opened = False

    def __repr__(self):
        return '<Context>'
