"""
The purpose of this module is to make all needed stuff for starting daemon.
This includes initialization of our Context with DaemonContext inside.

During context initialization we will close file descriptors, setup
signal handlers and so on.

It is important what before context initialization gevent will NOT be imported.
"""

from __future__ import absolute_import, print_function, division

import argparse
import os
import py
import sys
import yaml

from kernel.util.errors import formatException

from library import config

from .context import Context
from .. import PROGNAME


def parseArgs():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: (
            argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs)
        ),
        description='Run daemon.'
    )
    parser.add_argument('-d', '--daemonize', action='store_true')
    parser.add_argument('-b', '--background', action='store_true')
    parser.add_argument(
        '--workdir',
        help='Work directory (e.g. /Berkanavt/supervisor/var/skybone-coord)'
    )
    parser.add_argument(
        '--appdir', required=True,
        help='App directory (e.g. /Berkanavt/supervisor/services/skybone-coord)'
    )
    parser.add_argument(
        '-v', '--verbose', metavar='N', nargs='?', const=1, type=int, default=0,
        help='Increase verbosity by N times.'
    )
    parser.add_argument(
        '--config',
        help='Path to config file (if not set, library.config will be used)'
    )
    parser.add_argument(
        '-C', action='append', metavar='KEY=VALUE', default=[],
        help='Override config (can be specified several times)'
    )
    parser.add_argument('--trace-config', action='store_true', help='Trace configuration to stdout')

    parser.add_argument('--routers', type=int, default=None, required=False)
    parser.add_argument('--shards', type=int, default=None, required=False)
    parser.add_argument('--log-path', default=None, required=False)

    args = parser.parse_args()

    if args.C:
        cfg = {}

        for _ in args.C:
            key, value = _.split('=', 1)
            cfg[key] = yaml.load(value, Loader=yaml.FullLoader)

        args.cfg = cfg
    else:
        args.cfg = None

    return args


class SlottedDict(dict):
    def __getattr__(self, key):
        value = self[key]
        if isinstance(value, dict):
            value = self[key] = type(self)(value)
        self.__dict__[key] = value
        return value

    def __setattr__(self, key, value):
        self[key] = self.__dict__[key] = value


def load_config(path=None, overrides=None):
    if path is None:
        cfg = config.query('skynet.services.skybone-coord', overrides=overrides or {})
    else:
        assert not overrides
        cfg = SlottedDict(yaml.load(open(path, 'rb'), Loader=yaml.FullLoader))
    return cfg


def main():
    # Initialize daemon context
    try:
        ctx = Context()

        args = parseArgs()

        if not args.cfg:
            args.cfg = {}

        if args.routers is not None:
            args.cfg['main.routers'] = args.routers
        if args.shards is not None:
            args.cfg['main.shards'] = args.shards
         
        cfg = load_config(path=args.config, overrides=args.cfg)
        
        if args.log_path is not None:
            cfg['logpath'] = args.log_path
            
        if args.trace_config:
            __import__('pprint').pprint(cfg, stream=sys.stderr)
            raise SystemExit(0)

        ctx.initialize(
            progname=PROGNAME,
            background=False,
            close_stdio=False,
            print_hack=True,
            logging=True,
            logging_verbosity=args.verbose,
            logpath=args.log_path
        )
        ctx.cfg = cfg
        ctx.workdir = py.path.local(args.workdir)
        ctx.appdir = py.path.local(args.appdir)
    except SystemExit:
        raise
    except BaseException:
        sys.stderr.write(
            'ERROR (initializing): {0}\n'.format(formatException())
        )
        os._exit(1)  # pylint: disable=W0212

    try:
        with ctx:
            from ..main import main as realMain
            return realMain(ctx)
    except SystemExit:
        raise
    except BaseException:
        ctx.log.critical('Unhandled error: %s' % (formatException(), ))
        os._exit(1)  # pylint: disable=W0212


if __name__ == '__main__':
    raise SystemExit(main())
