import collections
import logging
import os
import threading
import time


class StatsDescriptor(object):
    __slots__ = 'label', 'type', 'unit', 'frames'

    def __init__(self, label, type, unit, frames):
        self.label = label
        self.type = type
        self.unit = unit
        self.frames = frames


class StatsFrame(object):
    frame_aliases = {
        1: 'now',
        60: '1m avg',
        300: '5m avg',
        3600: '1h avg',
    }

    def __init__(self):
        self.log = logging.getLogger('stats')
        self.data = collections.OrderedDict()

        self.sections = {}
        self.thread = None
        self._stop_flag = threading.Event()
        self._ts = 0

    def start(self):
        assert self.thread is None, 'already started!'
        self.thread = threading.Thread(target=self.loop)
        self.thread.start()
        return self

    def stop(self):
        assert self.thread is not None
        self._stop_flag.set()
        return self

    def join(self):
        self.thread.join(timeout=60)
        if self.thread.isAlive():
            self.log.critical('Stats loop failed to exit!')
            os._exit(1)
        return self

    def loop(self):
        while not self._stop_flag.isSet():
            nextrun = time.time() + 1
            self._gather_stats()
            self._stop_flag.wait(max(0, nextrun - time.time()))

    def add_section(self, name, provider):
        try:
            desc = [StatsDescriptor(*row) for row in provider(describe=True)]

            for descriptor in desc:
                if descriptor.label not in self.data:
                    self.data[descriptor.label] = (collections.deque(maxlen=max(descriptor.frames)), descriptor)

        except Exception:
            desc = []

        self.sections[name] = [provider, desc]

    def _gather_stats(self):
        for section_name, (provider, descs) in self.sections.iteritems():
            stats = provider()

            for idx, desc in enumerate(descs):
                dq = self.data[desc.label][0]
                dq.appendleft(stats[idx])

        self.get_stats()

    def get_stats(self):
        stats = {}
        for section_name, (_, descs) in self.sections.iteritems():
            section_stats = stats[section_name] = collections.OrderedDict()
            for desc in descs:
                desc_stats = section_stats[desc.label] = collections.OrderedDict()
                if desc.type == 'counter':
                    for frame in desc.frames:
                        frame_label = self.frame_aliases.get(frame, str(frame) + 's')
                        dq = self.data[desc.label][0]

                        value_now = dq[0]
                        value_start = dq[min((frame, len(dq) - 1))]
                        value = int((value_now - value_start) / frame)

                        desc_stats[frame_label] = (value, frame, desc.unit)
                elif desc.type == 'total':
                    assert desc.frames == [1]
                    dq = self.data[desc.label][0]
                    if len(dq) == 0:
                        value = 0
                    else:
                        value = dq[0]
                    desc_stats['now'] = (value, 1, desc.unit)

        return stats
