#!/usr/bin/env python

from __future__ import print_function

from ez_setup import use_setuptools
use_setuptools()

import os

del os.link

from setuptools import setup, find_packages, Extension
from setuptools.command.test import test as TestCommand          # noqa
from setuptools.command.sdist import sdist as SourceDistCommand  # noqa

from Cython.Build import cythonize


class PyTestCommand(TestCommand):
    user_options = [
        ('opts=', 'o', 'Options for pytest'),
    ]

    def initialize_options(self):
        self.opts = ''

    def finalize_options(self):
        import shlex
        self.opts_str = self.opts
        self.opts = shlex.split(self.opts)

    def run(self):
        if self.distribution.install_requires:
            self.distribution.fetch_build_eggs(self.distribution.install_requires)
        if self.distribution.tests_require:
            self.distribution.fetch_build_eggs(self.distribution.tests_require)

        self.with_project_on_sys_path(self.run_pytest)

    def run_pytest(self):
        import sys
        import pytest
        import os

        if os.fork():
            status = os.wait()[1]
            if os.WIFEXITED(status):
                raise SystemExit(os.WEXITSTATUS(status))
            else:
                self.announce('Killed by signal: %d' % (os.WSTOPSIG(status), ))
                raise SystemExit(1)
        else:
            sys.argv[:] = ['setup.py test'] + self.opts
            self.announce('Running pytest %s' % (self.opts_str, ))
            raise SystemExit(pytest.main())


class MySourceDistCommand(SourceDistCommand):
    def make_distribution(self):
        import os
        import textwrap

        for folder in 'tools', 'tests':
            for path, dirs, files in os.walk(folder):
                for filepath in files:
                    if filepath.endswith('.pyc'):
                        continue
                    self.filelist.append(os.path.join(path, filepath))

        open('setup.py', 'wb').write(textwrap.dedent('''
            import os
            import sys
            sys.path.insert(0, 'tools')
            execfile('tools/setup.py')
        ''').strip())

        self.filelist.append('setup.py')

        try:
            return SourceDistCommand.make_distribution(self)
        finally:
            os.unlink('setup.py')
            for f in self.filelist.files:
                if f.endswith('.pyx') and f.startswith('src/'):
                    os.unlink(f)

requirements = [
    req.strip() for req in open('tools/requirements.txt', 'rb').readlines()
    if (
        not req.startswith('#') and
        not req == '\n'
    )
]

extensions = []

for path, dirs, files in os.walk('src'):
    if not path.endswith('/'):
        path += '/'
    for fn in files:
        if fn == '__init__.py':
            continue
        if fn.endswith('.py'):
            modname = path.replace('src/', 'skybone_coord/').replace('/', '.') + fn[:-3]
            extensions.append(Extension(modname, [os.path.join(path, fn)]))

setup(
    name='skybone-coord',
    version='dev',
    description='Skybone Coordination Service',
    author='Vadim Fint',
    author_email='mocksoul@yandex-team.ru',
    zip_safe=False,
    install_requires=requirements,
    packages=[
        'skybone_coord'
    ] + [
        'skybone_coord.%s' % p for p in find_packages('src')
    ],
    package_dir={'skybone_coord': 'src'},
    ext_modules=cythonize(extensions),
    entry_points={
        'console_scripts': [
            'skybone-coord = skybone_coord.daemon.daemon:main',
            'skybone-coord-shard = skybone_coord.shard:main',
            'skybone-coord-router = skybone_coord.router:main',
        ]
    },
    cmdclass={
        'test': PyTestCommand,
        'sdist': MySourceDistCommand,
    },
)
