import argparse
import msgpack
import random
import socket

from lib.dc import DC, determine_dc


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--host', default='localhost', help='tracker host')
    parser.add_argument('--port', type=int, default=2399, help='tracker port')
    parser.add_argument('--uid', type=str, default='test_uid', help='peer uid')
    parser.add_argument('--resid', type=str, default='2bf2cd1c7ea39249b73bb19beac1fe782c3b79aa', help='resource id')
    return parser.parse_args()


def generate_tid():
    return int(random.random() * 2 ** 32 - 1)


def generate_connect_packet(uid, tid):
    pkt = msgpack.dumps((
        4079332072,  # initial cid
        0,           # action connect
        tid,
        uid,
        [
            '127.0.0.1', '::1',
            '127.0.0.2', '::2',
        ],
        12345,
        ['peer_types'],  # extensions
    ))
    return pkt


def generate_announce_packet(uid, tid, resid):
    pkt = msgpack.dumps((
        4079332072,  # initial cid
        10,          # action announce
        tid,
        uid,
        resid,
        1,           # announce downloading  (1 - dl, 2 - seed, 3 - stop)
        0,           # network - auto
    ))
    return pkt


def request(sock, addr, pkt):
    sock.sendto(pkt, addr)
    data = sock.recvfrom(2048)[0]
    return msgpack.loads(data)


def log_peers(peers):
    peers_by_dc = {dc: [] for dc in DC}
    for peer in peers:
        ip = peer['ips'][0]
        dc = determine_dc(ip)
        peers_by_dc[dc].append(ip)
    print('\n'.join(
        '{} in {}'.format(len(peers_by_dc[dc]), dc)
        for dc in sorted(peers_by_dc.keys(), key=lambda x: x.value)
        if peers_by_dc[dc]
    ))


def main():
    args = parse_args()

    host = socket.getaddrinfo(args.host, 0, socket.AF_INET6, socket.SOCK_DGRAM)[0][4][0]
    port = args.port
    addr = (host, port)

    uid = args.uid
    tid = generate_tid()
    resid = args.resid

    sock = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)

    pkt = generate_connect_packet(uid, tid)
    print('Send connect pkt')
    response = request(sock, addr, pkt)
    print('Connect response: %r\n' % response)

    pkt = generate_announce_packet(uid, tid, resid)
    print('Send announce pkt')
    response = request(sock, addr, pkt)
    print('Announce response: %r\n' % response)

    log_peers(response[5])

if __name__ == '__main__':
    main()
