import os


class BaseCtl(object):
    _server = None
    _cfg_section = None
    _name = None
    _libname = None

    def _rpc_client(self):
        raise NotImplementedError()

    def server(self):
        if not self._server:
            self._require()
            rpc_client = self._rpc_client()
            self._server = rpc_client(self.sock(), None)

        return self._server

    def _require(self, api=False):
        import sys
        import types
        import pkg_resources

        for name, path in (
            ('ya', None),
            ('ya.skynet', None),
            ('ya.skynet.services', None),
            ('ya.skynet.services.%s' % (
                self._name.replace('-', '_'),
            ), self.resolvePath('lib/%s' % (self._libname, ))),
        ):
            mod = sys.modules.setdefault(name, types.ModuleType(name))
            modpath = mod.__dict__.setdefault('__path__', [])
            if path and path not in modpath:
                modpath.append(path)

            if '.' in name:
                base, this = name.rsplit('.', 1)
                setattr(sys.modules[base], this, mod)

            pkg_resources.declare_namespace(name)

    def _config_dict(self):
        from library.config import query
        conf = query(self._cfg_section, allow_update=False, as_dict=True)
        return conf

    def _config(self, conf, read=False):
        import yaml
        import os

        etc_path = self.resolvePath('etc')
        if os.path.exists(etc_path) and not os.path.isdir(etc_path):
            os.unlink(etc_path)

        if not os.path.exists(etc_path):
            # This will create etc/ if it does not exists
            # But will not create whole services/copier dir in case of copier
            # service already removed from srvmngr
            os.mkdir(etc_path)

        conf_yaml = open(self.resolvePath('etc/config.yaml'), 'wb' if not read else 'rb')
        if read:
            return yaml.load(conf_yaml)
        else:
            yaml.dump(conf, conf_yaml, default_flow_style=False)

    def _update_config(self):
        try:
            stored = self._config(None, read=True)
        except:
            stored = None

        actual = self._config_dict()

        if actual != stored:
            import os
            if os.uname()[0].lower() == 'darwin' or os.uname()[0].lower().startswith('cygwin'):
                has_root = os.getuid() == 0
            else:
                has_root = os.getresuid()[2] == 0

            if has_root:
                from kernel.util.sys.user import userPrivileges
                with userPrivileges():
                    self._config(actual)
            else:
                self._config(actual)

    def _workdir(self):
        import os
        try:
            from api.srvmngr import getRoot
        except ImportError:
            # TODO: remove after 12.x skynet will spread the world  # noqa
            from srvmngr.utils import getRoot

        if self._name == 'copier':
            return os.path.join(getRoot(), 'var', self._name, 'rbtorrent')
        else:
            return os.path.join(getRoot(), 'var', self._name)

    def sock(self):
        import os
        if 'SKYBONE_RPC_SOCK' in os.environ:
            return os.environ['SKYBONE_RPC_SOCK']
        return os.path.join(self._workdir(), 'rpc.sock')

    def start(self):
        raise NotImplementedError()

    def stop(self):
        try:
            self.server().call('stop').wait()
        except:
            # If we cant import copier libraries somehow -- try to send simple SIGINT signal
            # to all processes we have.
            for pp in self.spawnedProcesses():
                pp.send_signal('SIGINT')

    def running(self):
        raise NotImplementedError()

    def api(self):
        raise NotImplementedError()

    def name(self):
        return self._name

    def cpuAffinity(self):  # noqa
        # Request sticky CPU group named "copier" of 2 CPUs.
        return self.name(), 2

    def depends(self):
        return []


class SkyboneCtl(BaseCtl):
    _name = 'copier'
    _libname = 'skybone'
    _cfg_section = 'skynet.services.copier'

    def _rpc_client(self):
        from ya.skynet.services.copier.rpc.client import RPCClient
        return RPCClient

    def port(self):
        from api import config
        return config.basePortOffset() + 10018

    def data_port(self):
        from api import config
        return config.basePortOffset() + 6881

    def start(self):
        import os

        if os.uname()[0].lower() == 'darwin' or os.uname()[0].lower().startswith('cygwin'):
            has_root = os.getuid() == 0
        else:
            has_root = os.getresuid()[2] == 0

        if has_root:
            user = 'root'
        else:
            user = None

        self._update_config()

        # Export PYTHONPATH plus library path
        python_path = os.environ.get('PYTHONPATH', '').split(':')
        python_path.insert(0, self.resolvePath('lib'))
        environ = {
            'PYTHONPATH': ':'.join(python_path),
        }

        # Export all SKYNET* environment variables
        for key, value in os.environ.items():
            if key.startswith('SKYNET'):
                environ[key] = value

        kwargs = dict(
            env=environ,
            afterlife=0,
            logStderr=True,
            liner=True
        )

        if user:
            kwargs['user'] = user

        self.createProcess(
            [
                self.resolvePath('bin/skybone'),
                '-c', self.resolvePath('etc/config.yaml'),
                '-d', self._workdir(),
            ] + (['-u', 'skynet'] if user else []),
            **kwargs
        )

    def running(self):
        self._update_config()

        from api.logger import SkynetLoggingHandler
        import logging

        handler = SkynetLoggingHandler(app='skybone')
        log = logging.getLogger('ctl')
        log.setLevel(logging.DEBUG)
        log.addHandler(handler)
        log.propagate = False

        log.info('(start) running')

        import errno
        import time

        try:
            result = self.server().call('ping', extended=True).wait()
            log.debug('ping result: %r', result)
        except Exception as ex:
            log.warning('ping error: %s', ex)
            raise

        if result['stage'][1] != 'done':
            in_stage = time.time() - result['stage'][0]
            if in_stage < 86400:
                log.debug(
                    'just %d secs in %s stage, bypassing all checks',
                    time.time() - result['stage'][0],
                    result['stage'][1]
                )
                return True
            else:
                log.debug(
                    '%d secs in %s stage, too long, reporting as running=False',
                    time.time() - result['stage'][0],
                    result['stage'][1]
                )
                return False

        active = result.get('active', False)
        if not active:
            log.debug('not active, running=False')
            return False

        import socket

        def _check_sock(name, addr, family=socket.AF_INET):
            log.debug('check %s socket', name)

            try:
                sock = socket.socket(family, socket.SOCK_STREAM)
                sock.settimeout(10)
                sock.connect((addr, self.data_port()))
            except socket.error as ex:
                log.warning('unable to connect to %s socket: %s, running=False', name, ex)
                return False
            except Exception as ex:
                log.warning('unknown error while connecting to %s sockect: %s, running=False', name, ex)
                return False
            else:
                try:
                    sock.send('a' * (49 + 19))
                    data = sock.recv(1)
                    if data != '':
                        log.warning(
                            '%s socket didnt closed connection (result: %r), running=False',
                            name, data
                        )
                        return False  # copier must close connection asap
                except socket.error as ex:
                    if ex.errno != errno.ECONNRESET:  # con reset by peer
                        log.debug('%s socket not good error: %s, running=False', name, ex)
                        return False
                except Exception as ex:
                    log.debug('%s socket unknown error: %s, running=False', name, ex)
                    return False

                log.debug('%s sockect ok', name)
                sock.close()

                return True

        if not _check_sock('bb', '127.0.0.1'):
            return False

        if result.get('fastbone_mode') is True:
            if not _check_sock('fb', '::1', family=socket.AF_INET6):
                return False
        else:
            log.debug('bypassing fb socket check')

        log.debug('(finish) runing: True')
        return True

    def api(self):
        self._require()
        from ya.skynet.services.copier.client.proxy import Copier
        return Copier


class SkyboneMDSCtl(BaseCtl):
    _name = 'skybone-mds'
    _libname = 'skybone_mds'
    _cfg_section = 'skynet.services.skybone-mds'

    def _rpc_client(self):
        from ya.skynet.services.skybone_mds.rpc.client import RPCClient
        return RPCClient

    def start(self):
        self._update_config()

        # Export PYTHONPATH plus library path
        python_path = os.environ.get('PYTHONPATH', '').split(':')
        python_path.insert(0, self.resolvePath('lib'))
        environ = {
            'PYTHONPATH': ':'.join(python_path),
        }

        # Export all SKYNET* environment variables
        for key, value in os.environ.items():
            if key.startswith('SKYNET'):
                environ[key] = value

        kwargs = dict(
            env=environ,
            afterlife=0,
            logStderr=True,
            liner=True
        )

        self.createProcess(
            [
                self.resolvePath('bin/skybone-mds'),
                '-c', self.resolvePath('etc/config.yaml'),
                '-d', self._workdir(),
            ],
            **kwargs
        )

    def running(self):
        import time

        try:
            result = self.server().call('ping', extended=True).wait()
        except Exception:
            raise

        if result['stage'][1] != 'done':
            in_stage = time.time() - result['stage'][0]
            if in_stage < 900:
                return True
            else:
                return False

        active = result.get('active', False)
        if not active:
            return False

        return True

basename = os.path.basename(os.path.dirname(__file__))

if basename in ('copier', 'skybone'):
    Ctl = SkyboneCtl
elif basename == 'skybone-mds':
    Ctl = SkyboneMDSCtl
else:
    # default
    Ctl = SkyboneCtl
