-- Main resource and file storage tables

-- resource.id is a sha1 hash of:
-- lt.bencode({'length': 123, 'name': 'metadata', 'piece length': 4194304, 'pieces': <sha1piecehashes>})

-- data.sha1 is a full data sha1 hash

CREATE TABLE resource_type (
    type            TEXT    NOT NULL,
    PRIMARY KEY (type) ON CONFLICT ABORT
    CHECK (type IN ('rbtorrent1'))
) WITHOUT ROWID;

INSERT INTO resource_type VALUES ("rbtorrent1");

-- Each resource constist of:
-- 1. [data][type=rbtorrent1] resource data, which constists of 2 parts:
--    * torrent_info to download part #2
--    * resource structure msgpacked

--    Example:
--    MsgPack({
--       'rbtorrent1': MsgPack({
--           'structure': {'path/to/file': <data_description>}
--           'torrents': {<infohash>: <torrent_info>},
--        }),
--       'torrent_info': {
--           'name': 'metadata',
--           'piece length': <piece_size>,
--           'pieces': <pieces>,
--           'length': <size of rbtorrent1 binary above>
--       }
--    })
--
--    Both of them is generated only once and then downloaded by
--    any peers interested in resource.
--
-- 2. [id][type=rbtorrent1] basically this is sha1 hash of torrent_info in data.
--    At least for rbtorrent1 resources.
CREATE TABLE resource (
    "id"            TEXT    NOT NULL,   -- resource identifier (sha1 hash)
    "type"          TEXT    NOT NULL,   -- resource_type.type
    "data"          BLOB    NOT NULL,   -- resource data (see above)
    "torrents_count" INTEGER NOT NULL,  -- count of unique torrents in resource (used by announcer)

    PRIMARY KEY (id)        ON CONFLICT ABORT
    FOREIGN KEY (type)      REFERENCES resource_type ON DELETE RESTRICT ON UPDATE RESTRICT
);

CREATE TABLE data (
    "id"            TEXT    NOT NULL,  -- uid (<hash>)
    "legacy_id"     BOOL    NOT NULL,  -- True if id is md5, False for sha1
    "size"          INTEGER NOT NULL,  -- data size
    "md5"           TEXT    NOT NULL,  -- md5
    "sha1_blocks"   BLOB    NOT NULL,  -- sha1 block hashes in msgpack [version, [[size, hash], [size, hash], ...]]

    PRIMARY KEY (id)        ON CONFLICT ABORT
    UNIQUE      (md5)
);

CREATE TRIGGER "check-legacy-md5-trigger" BEFORE INSERT ON data BEGIN
    SELECT CASE (new.legacy_id == (new.id == new.md5))
        WHEN 0 THEN RAISE (ABORT, "id != md5, but legacy_id set 1")
    END;
END;

CREATE TABLE file (
    "path"          TEXT    NOT NULL,  -- path (absolute, but not real)
    "data"          TEXT    NOT NULL,  -- data.id
    "mtime"         INTEGER NOT NULL,  -- inode mtime
    "chktime"       INTEGER NOT NULL,  -- last checking time

    PRIMARY KEY (path)      ON CONFLICT ABORT
    FOREIGN KEY (data)      REFERENCES data           ON DELETE CASCADE ON UPDATE RESTRICT
);

CREATE INDEX idx__file__data ON file(data);
CREATE INDEX idx__file__chktime ON file(chktime);

CREATE TABLE resource_data (
    "resource"      TEXT    NOT NULL,  -- resource.id
    "name"          TEXT    NOT NULL,  -- name of file
    "data"          TEXT        NULL,  -- data.id
    "symlink"       TEXT        NULL,  -- symlink target
    "directory"     BOOL    NOT NULL,  -- directory target
    "perms"         INT     NOT NULL,  -- permissions (e.g. 0o640)

    PRIMARY KEY (resource, name) ON CONFLICT ABORT

    FOREIGN KEY (resource)  REFERENCES resource       ON DELETE CASCADE ON UPDATE RESTRICT
    FOREIGN KEY (data)      REFERENCES data           ON DELETE CASCADE ON UPDATE RESTRICT

    CONSTRAINT "exclusive-data-symlink-directory" CHECK (
        CASE (data IS NOT NULL)
            WHEN 1 THEN (symlink IS NULL AND NOT directory)
            WHEN 0 THEN CASE (symlink IS NOT NULL)
                WHEN 1 THEN (NOT directory)
                WHEN 0 THEN (directory)
            END
        END
    )
);

CREATE INDEX idx__resource_data__resource ON resource_data (resource);
CREATE INDEX idx__resource_data__data ON resource_data (data);
