def migrate(db, log):
    import msgpack
    from collections import defaultdict

    def genrb(resid, structure, torrents):
        import hashlib
        from .bencode import bencode

        data = bencode({'structure': structure, 'torrents': torrents})

        piece_size = 4 * 1024 * 1024
        num_pieces = len(data) // piece_size
        if len(data) % piece_size:
            num_pieces += 1

        pieces = []
        for idx in xrange(num_pieces):
            start = idx * piece_size
            end = start + piece_size

            piece_data = data[start:end]
            pieces.append(hashlib.sha1(piece_data).digest())

        rb = {
            'name': 'metadata',
            'piece length': piece_size,
            'pieces': ''.join(pieces),
            'length': len(data)
        }

        if hashlib.sha1(bencode(rb)).hexdigest() == resid:
            return rb
        else:
            raise Exception('Generated RB data resource id does not match')

    restypes = defaultdict(int)
    counts = {
        'bad_msgpack': 0,
        'no_torrent_info': 0,
        'converted': 0,
        'converted_fail': 0,
        'ok': 0
    }

    total = db.query_one_col('SELECT COUNT(*) FROM resource', log=False)
    log.info('  Migrating %d resources...', total)
    for idx, (resid, typ, data) in enumerate(db.query('SELECT id, type, data FROM resource', log=False)):
        if idx % 1000 == 0 and idx != 0:
            log.debug('    Fixed %d of %d resources', idx, total)

        restypes[typ] += 1
        if typ != 'rbtorrent1':
            continue

        try:
            data = msgpack.loads(data)
        except Exception as ex:
            log.warning('    Bad data msgpack for resource %s: %s', resid, ex)
            counts['bad_msgpack'] += 1
            continue

        if 'torrent_info' not in data:
            counts['no_torrent_info'] += 1

        if data['torrent_info'] is None:
            try:
                data['torrent_info'] = genrb(resid, data['structure'], data['torrents'])
                db.query('UPDATE resource SET data = ? WHERE id = ?', (buffer(msgpack.dumps(data)), resid), log=False)
                counts['converted'] += 1
            except Exception as ex:
                log.warning('    Unable to convert ti for resource %s: %s', resid, ex)
                counts['converted_fail'] += 1
                continue
        else:
            counts['ok'] += 1
    log.info('  done')
    log.info('  Resource types:')
    for restype, count in restypes.iteritems():
        log.info('    %s: %d', restype, count)

    log.info('  Counts:')
    for t, count in counts.iteritems():
        log.info('    %s: %d', t, count)
