def migrate(db, log):
    db.query('DROP TABLE IF EXISTS file_new')
    db.query('''\
CREATE TABLE file_new (
    "path"          TEXT    NOT NULL,  -- path
    "inode"         INTEGER NOT NULL,  -- inode
    "data"          TEXT    NOT NULL,  -- data.id
    "mtime"         INTEGER NOT NULL,  -- inode mtime
    "chktime"       INTEGER NOT NULL,  -- last checking time

    PRIMARY KEY (path)      ON CONFLICT ABORT
    FOREIGN KEY (data)      REFERENCES data           ON DELETE CASCADE ON UPDATE RESTRICT
);''')

    from skybone.rbtorrent.utils import Path

    total_files = db.query_one_col('SELECT COUNT(*) FROM file')
    converted_files = 0
    failed_files = 0
    missing_data = 0

    log.info('  starting file table migrate. Total %d files', total_files)

    datas = set(db.query_col('SELECT id FROM data'))

    # This can use a lot of memory, but there is no way to do same differently :(
    for path, data, mtime, chktime in db.query(
        'SELECT path, data, mtime, chktime FROM file'
    ):
        if data not in datas:
            missing_data += 1
            continue

        try:
            path = Path(path)
            stat = path.stat()
        except Exception as ex:
            log.warning('  %s: unable to stat: %s: %s', path, type(ex).__name__, ex)
            failed_files += 1
            continue

        inode = stat.ino

        db.query(
            'INSERT INTO file_new (path, inode, data, mtime, chktime) VALUES (?, ?, ?, ?, ?)',
            [path.strpath, inode, data, mtime, chktime],
            log=False
        )

        converted_files += 1

        if converted_files % 1000 == 0:
            percent = float(converted_files) / total_files * 100
            log.debug(
                '  [%d%%] %d/%d files done (%d failed, %d missing data)',
                percent, converted_files, total_files,
                failed_files, missing_data
            )

    log.info('  finished. Converted %d from %d files, %d files failed', converted_files, total_files, failed_files)

    db.query('DROP TABLE file')
    db.query('ALTER TABLE file_new RENAME TO file')

    db.query('CREATE INDEX idx__file__chktime ON file(chktime)')
    db.query('CREATE INDEX idx__file__inode ON file(inode)')
    db.query('CREATE INDEX idx__file__data ON file(data)')
