"""
The purpose of this module is to perform administrative (maintenance) tasks on running copier instance.
"""

from __future__ import absolute_import, print_function, division

import argparse
import locale
import msgpack
import sys
import time

from ..kernel_util.functional import singleton
from ..rbtorrent import logger as logging
from ..rbtorrent.utils import human_size_sep, human_time
from ..rpc.errors import CallError, CallFail


@singleton
def log():
    return logging.initialize_client()


def _server():
    from api.skycore import ServiceManager
    from api.skycore.errors import NamespaceError
    try:
        return ServiceManager().get_service_python_api('skynet', 'skybone', 'python_rpc')
    except NamespaceError:
        pass


def retry_call(name, *args, **kwargs):
    tries = 0
    while True:
        tries += 1
        try:
            result = _server().call(name, *args, **kwargs).wait()
        except Exception as ex:
            if isinstance(ex, (CallError, CallFail)):
                raise

            if tries >= 10:
                raise

            log().debug('admin (%s): server.call: error: %s, tries: %d', name, ex, tries)
            time.sleep(min(10, tries * 2))
        else:
            return result


def _formatter(data, fmt):
    formatters = {
        'text': lambda x: repr(x),
        'yaml': lambda x: __import__('yaml').dump(x),
        'json': lambda x: __import__('json').dumps(x),
        'pprint': lambda x: __import__('pprint').pformat(x),
        'msgpack': lambda x: __import__('msgpack').dumps(x),
    }
    print(formatters[fmt](data))
    return 0


def resource_info(args):
    import subprocess as subproc
    from library.config import query
    import random
    import socket
    import gevent

    coord_hosts = query('skynet.services.copier')['coord_hosts']
    coord_port = query('skynet.services.copier')['coord_hosts_port']

    if isinstance(coord_hosts, (list, tuple)):
        hosts = list(coord_hosts)
    else:
        proc = subproc.Popen(['/skynet/tools/sky', 'list', coord_hosts], stdout=subproc.PIPE)
        proc.wait()
        assert proc.returncode == 0

        hosts = proc.stdout.read().strip().split()

    sock = gevent.socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)

    requests = {}
    results = {}

    resid = args.resid.split(':', 1)[1]

    for host in hosts:
        req_id = random.randint(0, 2 ** 32 - 1)

        try:
            sock.sendto(msgpack.dumps((
                'IN', 'FO',
                resid.decode('hex'),
                req_id
            )), (host, coord_port))
        except:
            continue
        else:
            requests[req_id] = host

    if args.wait_trackers:
        deadline = args.wait_trackers + time.time()
    else:
        deadline = None

    while True:
        if not requests:
            break

        if not args.wait_trackers and len(results) >= 2:
            break

        if deadline:
            wait = deadline - time.time()
            if wait <= 0:
                break

            try:
                gevent.socket.wait_read(sock.fileno(), wait)
            except:
                break

        data, peer = sock.recvfrom(8192)
        data = msgpack.loads(data)

        req = requests.pop(data['req'])
        results[req] = data

        if 'atime' not in data:
            data['atime'] = 0

    resinfo = {'seeders': 0, 'leechers': 0, 'atime': 0}
    residbin = resid.decode('hex')

    for host, info in results.iteritems():
        if residbin not in info['info']:
            continue

        ri = info['info'][residbin]

        if 'atime' not in ri:
            # Only newer skybone-coords support reporting atime
            ri['atime'] = 0

        resinfo['seeders'] = max(ri['seeders'], resinfo['seeders'])
        resinfo['leechers'] = max(ri['leechers'], resinfo['leechers'])
        resinfo['atime'] = max(ri['atime'], resinfo['atime'])

    return _formatter({args.resid: resinfo}, args.format)


def status(args):
    status = retry_call('status', dbwait=args.wait)
    if args.format != 'text':
        return _formatter(status, args.format)

    from collections import defaultdict

    class NotFound(object):
        def __str__(self):
            return '<???>'

        def __getitem__(self, key):
            return NotFound()

        __repr__ = __str__

    pstatus = defaultdict(NotFound)

    def grab_value(path, default=NotFound(), fltr=str):
        if isinstance(path, (list, tuple)):
            return [grab_value(onepath, default, fltr=fltr) for onepath in path]

        parts, key = path.rsplit('.', 1)
        parts = parts.split('.')

        # (root, key, item)
        stack = [(None, None, status)]
        value = default

        for part in parts:
            parent = stack[-1][2]
            current = parent.get(part, None)

            if not isinstance(current, dict):
                break

            stack.append((parent, part, current))
        else:
            value = stack[-1][2].pop(key, value)
            if value != default:
                value = fltr(value)

        for (parent, key, item) in reversed(stack):
            if not item and parent is not None:
                del parent[key]
            else:
                break

        return value

    # Daemon section
    stopping, active = grab_value((
        'daemon.stopping', 'daemon.active'
    ), False)

    if stopping:
        pstatus['status'] = 'stopping'
    elif active:
        pstatus['status'] = 'active'
    else:
        pstatus['status'] = 'unknown'

    daemon_uptime = grab_value('daemon.uptime', fltr=int)
    pstatus['uptime'] = human_time(daemon_uptime)
    pstatus['uid'] = grab_value('db.sqlite.maintainance.uid')

    if daemon_uptime > 0:
        cpu_self = sum(grab_value((
            'daemon.rusage.self.user',
            'daemon.rusage.self.system',
        ), fltr=float)) / float(daemon_uptime)

        cpu_children = sum(grab_value((
            'daemon.rusage.children.user',
            'daemon.rusage.children.system',
        ), fltr=float)) / float(daemon_uptime)
    else:
        status['daemon'].pop('rusage')
        cpu_self = cpu_children = 0

    pstatus['cpu_percent_self'] = cpu_self
    pstatus['cpu_percent_children'] = cpu_children

    pstatus['greenlets_count'] = grab_value('daemon.greenlets')

    # Resource manager section
    pstatus['resource_mngr_gc_files_checked'] = grab_value('resource_mngr.gc.files_checked')
    pstatus['resource_mngr_gc_files_removed'] = grab_value('resource_mngr.gc.files_removed')
    pstatus['resource_mngr_gc_files_removed_bad_checksum'] = (
        grab_value('resource_mngr.gc.files_removed_by_bad_checksum')
    )
    pstatus['resource_mngr_gc_files_removed_bad_mtime'] = (
        grab_value('resource_mngr.gc.files_removed_by_bad_mtime')
    )

    # File cache
    pstatus['file_cache_resources_count'] = grab_value('file_cache.resources_count')
    pstatus['file_cache_data_count'] = grab_value('file_cache.data_count')
    pstatus['file_cache_files_count'] = grab_value('file_cache.files_count')
    pstatus['file_cache_wait_checking'] = grab_value('file_cache.wait_checking')
    pstatus['file_cache_data_size'] = grab_value('file_cache.data_size', fltr=human_size_sep)

    # Jobs section
    line_job_stat = ''
    job_stat_str = 'RPC:\n'
    job_stat_str += '  Jobs (in progress, finished, error, failed):'
    all_job_types = sorted(set(
        status['rpc']['active']['jobs'].keys() +
        status['rpc']['counters']['completed'].keys() +
        status['rpc']['counters']['errors'].keys() +
        ['resource_create', 'resource_download', 'resource_info']
    ))

    for job in all_job_types:
        job_active = status['rpc']['active']['jobs'].get(job, 0)
        job_successes = status['rpc']['counters']['completed'].get(job, 0)
        job_errors = status['rpc']['counters']['errors'].get(job, {})

        job_error = job_fatal = 0

        for err, count in job_errors.iteritems():
            if err in (
                'ApiError',
                'UnshareableResource',
                'ResourceDownloadError',
                'Timeout',
                'ResourceNotAvailable',
                'ResourceNotAllowedByNetwork',
                'FilesystemError',
            ):
                job_error += count
            else:
                job_fatal += count

        job_stat_str += '\n    %-20s: %s/%s/%s/%s' % (
            job,
            job_active,
            job_successes,
            job_error,
            job_fatal
        )

        if job == 'resource_create':
            short_job = 'cre'
        elif job == 'resource_download':
            short_job = 'dl '
        elif job == 'resource_info':
            short_job = 'inf'
        else:
            short_job = None

        if short_job:
            line_job_stat += '| %s %5s %5s %5s %5s ' % (
                short_job,
                job_active, job_successes,
                job_error, job_fatal
            )

    job_stat_str += '\n  Messages (total sessions: %d):' % (status['rpc']['counters']['sessions'], )

    for msg, count in status['rpc']['counters']['messages'].iteritems():
        job_stat_str += '\n    %-20s: %s' % (msg, count)

    status.pop('rpc')
    pstatus['job_stat'] = job_stat_str
    pstatus['line_job_stat'] = line_job_stat

    pstatus['jobs_in_progress'] = grab_value('jobs.pending')
    pstatus['jobs_success_count'] = grab_value('jobs.success')
    pstatus['jobs_failed_count'] = grab_value('jobs.failed')

    # Announcer section
    pstatus['announcer_pending_count'] = grab_value('announcer.pending')
    pstatus['announcer_packets_sent'] = grab_value('announcer.stats.packets.sent')
    pstatus['announcer_packets_recv'] = grab_value('announcer.stats.packets.recv')
    pstatus['announcer_packets_lost'] = grab_value('announcer.stats.packets.lost')

    pstatus['dlrs_skbt_ok'] = grab_value('skybit.stats.ok')
    pstatus['dlrs_skbt_err'] = grab_value('skybit.stats.error')

    pstatus['skbt_seed_mem_size'] = grab_value('skybit.seeder.memory.size', fltr=human_size_sep)
    pstatus['skbt_seed_mem_free'] = grab_value('skybit.seeder.memory.free', fltr=human_size_sep)

    # Database section
    locked = grab_value('db.sqlite.locked', fltr=bool)
    if locked:
        lock_reason = grab_value('db.sqlite.lock_reason')
        if lock_reason == 'None':
            lock_reason = 'unknown'
        pstatus['sqlite_locked_str'] = ', locked (reason: %s)' % (lock_reason, )
    else:
        pstatus['sqlite_locked_str'] = ''

    pstatus['sqlite_job_waiters'] = grab_value('db.sqlite.job_waiters', fltr=int)
    pstatus['sqlite_mem_generic'] = grab_value('db.sqlite.memory.generic', fltr=human_size_sep)
    pstatus['sqlite_mem_total'] = grab_value('db.sqlite.memory.total', fltr=human_size_sep)
    pstatus['sqlite_mem_cache'] = grab_value('db.sqlite.memory.cache', fltr=human_size_sep)
    pstatus['sqlite_cache_hit'] = grab_value('db.sqlite.cache.hit', fltr=int)
    pstatus['sqlite_cache_miss'] = grab_value('db.sqlite.cache.miss', fltr=int)
    pstatus['sqlite_cache_hit_miss_maxlen'] = max(
        len(str(_)) for _ in (pstatus['sqlite_cache_hit'], pstatus['sqlite_cache_miss'])
    )
    pstatus['sqlite_cache_writes'] = grab_value('db.sqlite.cache.writes', fltr=human_size_sep)
    pstatus['sqlite_prep_stmt_used'] = grab_value('db.sqlite.memory.prep_stmt', fltr=human_size_sep)

    data_used, data_free = grab_value((
        'db.sqlite.data.used_bytes', 'db.sqlite.data.free_bytes',
    ), fltr=int)

    if not isinstance(data_free, NotFound) and not isinstance(data_used, NotFound):
        data_free_percent = data_free / data_used
        pstatus['sqlite_data_used_bytes'] = human_size_sep(data_used)
        pstatus['sqlite_data_free_bytes'] = human_size_sep(data_free)
        pstatus['sqlite_data_free_percent'] = data_free_percent
    else:
        data_free_percent = NotFound()
        pstatus['sqlite_data_used_bytes'] = data_used
        pstatus['sqlite_data_free_bytes'] = data_free
        pstatus['sqlite_data_free_percent'] = 0

    try:
        total_cache_tries = pstatus['sqlite_cache_hit'] + pstatus['sqlite_cache_miss']
        pstatus['sqlite_cache_hit_percent'] = pstatus['sqlite_cache_hit'] / float(total_cache_tries)
        pstatus['sqlite_cache_miss_percent'] = pstatus['sqlite_cache_miss'] / float(total_cache_tries)
    except:
        pstatus['sqlite_cache_hit_percent'] = pstatus['sqlite_cache_miss_percent'] = 0

    last_vacuum = grab_value('db.sqlite.maintainance.last_vacuum', fltr=int)
    if not isinstance(last_vacuum, NotFound):
        pstatus['sqlite_maintainance_vacuum'] = human_time(time.time() - last_vacuum)

    pstatus['sqlite_maintainance_vacuum_duration'] = grab_value(
        'db.sqlite.maintainance.last_vacuum_duration', fltr=human_time
    )

    last_analyze = grab_value('db.sqlite.maintainance.last_analyze', fltr=int)
    if not isinstance(last_analyze, NotFound):
        pstatus['sqlite_maintainance_analyze'] = human_time(time.time() - last_analyze)

    pstatus['sqlite_maintainance_analyze_duration'] = grab_value(
        'db.sqlite.maintainance.last_analyze_duration', fltr=human_time
    )

    pstatus['sqlite_active_job'] = grab_value('db.sqlite.job', fltr=tuple)

    pstatus['proxy_bytes_in'] = grab_value('proxy.bytes_in', fltr=human_size_sep)
    pstatus['proxy_bytes_ou'] = grab_value('proxy.bytes_out', fltr=human_size_sep)
    pstatus['proxy_bytes_in_trnt'] = grab_value('proxy.bytes_in_torrent', fltr=human_size_sep)
    pstatus['proxy_bytes_in_skbt'] = grab_value('proxy.bytes_in_skybit', fltr=human_size_sep)
    pstatus['proxy_bytes_ou_trnt'] = grab_value('proxy.bytes_out_torrent', fltr=human_size_sep)
    pstatus['proxy_bytes_ou_skbt'] = grab_value('proxy.bytes_out_skybit', fltr=human_size_sep)
    pstatus['proxy_bytes_ou_direct'] = grab_value('proxy.bytes_out_direct', fltr=human_size_sep)

    pstatus['proxy_infohash_cache_size'] = grab_value('proxy.infohash_lookup.cache_size')
    pstatus['proxy_infohash_cache_hits'] = grab_value('proxy.infohash_lookup.cache_hits')
    pstatus['proxy_infohash_cache_miss'] = grab_value('proxy.infohash_lookup.cache_miss')
    pstatus['proxy_infohash_found'] = grab_value('proxy.infohash_lookup.found')
    pstatus['proxy_infohash_not_found'] = grab_value('proxy.infohash_lookup.missing')
    pstatus['proxy_infohash_time_spent'] = grab_value('proxy.infohash_lookup.spent', fltr=human_time)
    pstatus['proxy_main_connects'] = grab_value('proxy.incoming.connects')
    pstatus['proxy_main_connects_local'] = grab_value('proxy.incoming.local')
    pstatus['proxy_main_connects_fails'] = grab_value('proxy.incoming.connect_fails')
    pstatus['proxy_main_connects_noslot'] = grab_value('proxy.incoming.noslot_count')
    pstatus['proxy_main_connects_nohash'] = grab_value('proxy.incoming.no_infohash')
    pstatus['proxy_main_torrent_connects'] = grab_value('proxy.incoming.torrent.connects')
    pstatus['proxy_main_torrent_connects_fails'] = grab_value('proxy.incoming.torrent.connect_fails')
    pstatus['proxy_main_torrent_no_infohash'] = grab_value('proxy.incoming.torrent.no_infohash')
    pstatus['proxy_main_skybit_connects'] = grab_value('proxy.incoming.skybit.connects')
    pstatus['proxy_main_skybit_connects_fails'] = grab_value('proxy.incoming.skybit.connect_fails')
    pstatus['proxy_main_skybit_no_resource'] = grab_value('proxy.incoming.skybit.no_resource')

    pstatus['proxy_main_no_handshake'] = grab_value('proxy.incoming.invalid_handshake')

    pstatus['proxy_proxy_connects'] = grab_value('proxy.proxy.cmd_connect_cnt')
    pstatus['proxy_proxy_connect_fails'] = grab_value('proxy.proxy.connect_fail')
    pstatus['proxy_proxy_invalid_proto'] = grab_value('proxy.proxy.invalid_proto')

    pstatus['proxy_uds_connects'] = grab_value('proxy.uds.connects')
    pstatus['proxy_uds_connects_fails'] = grab_value('proxy.uds.connect_fails')

    import string

    proxy_status = '''\
  Traffic:  |            in               out
    all     |\
  {proxy_bytes_in[0]:>12} {proxy_bytes_in[1]}\
  {proxy_bytes_ou[0]:>12} {proxy_bytes_ou[1]}
    torrent |\
  {proxy_bytes_in_trnt[0]:>12} {proxy_bytes_in_trnt[1]}\
  {proxy_bytes_ou_trnt[0]:>12} {proxy_bytes_ou_trnt[1]}
    skybit  |\
  {proxy_bytes_in_skbt[0]:>12} {proxy_bytes_in_skbt[1]}\
  {proxy_bytes_ou_skbt[0]:>12} {proxy_bytes_ou_skbt[1]}
  HashCache:
    size      hits    misses     found  notfound      time
  {proxy_infohash_cache_size:>6}\
  {proxy_infohash_cache_hits:>8}\
  {proxy_infohash_cache_miss:>8}\
  {proxy_infohash_found:>8}\
  {proxy_infohash_not_found:>8}\
  {proxy_infohash_time_spent:>8}
  Stats:
    sock  |       connects        local        fails       nohash  nohandshake     protoerr
    main  |\
  {proxy_main_connects:>13}\
  {proxy_main_connects_local:>11}\
  {proxy_main_connects_fails:>11}\
  {proxy_main_connects_nohash:>11}\
  {proxy_main_no_handshake:>11}\
            -
 T  main  |\
  {proxy_main_torrent_connects:>13}\
            -\
  {proxy_main_torrent_connects_fails:>11}\
  {proxy_main_torrent_no_infohash:>11}\
            -\
            -
 S  main  |\
  {proxy_main_skybit_connects:>13}\
            -\
  {proxy_main_skybit_connects_fails:>11}\
  {proxy_main_skybit_no_resource:>11}\
            -\
            -
 T  proxy |\
  {proxy_proxy_connects:>13}\
            -\
  {proxy_proxy_connect_fails:>11}\
            -\
            -\
  {proxy_proxy_invalid_proto:>11}
 S  proxy |\
  {proxy_uds_connects:>13}\
            -\
  {proxy_uds_connects_fails:>11}\
            -\
            -\
            -
    noslt |\
  {proxy_main_connects_noslot:>13}\
            -\
            -\
            -\
            -\
            -\
'''.rstrip()

    pstatus['proxy_status'] = string.Formatter().vformat(proxy_status, (), pstatus)

    if not args.line:
        print(string.Formatter().vformat('''
Copier v2.0  (uid: {uid})

Status: {status}  |  uptime {uptime}  |  CPU {cpu_percent_self:2.0%} (+{cpu_percent_children:2.0%})

{job_stat}

Gevent
  Greenlets    : {greenlets_count}

ResourceMngr:
  FileChecker:
    GC:
      checked : {resource_mngr_gc_files_checked}
      removed : {resource_mngr_gc_files_removed}
    Direct:
      mtime   : {resource_mngr_gc_files_removed_bad_mtime}
      checksum: {resource_mngr_gc_files_removed_bad_checksum}

File cache
  Data size    : {file_cache_data_size[0]:>6} {file_cache_data_size[1]}
  Resources cnt: {file_cache_resources_count}
  Data cnt     : {file_cache_data_count}
  Files cnt    : {file_cache_files_count}
  Wait checking: {file_cache_wait_checking}

Announcer
  Pending      : {announcer_pending_count}
  Packets      : {announcer_packets_sent} / {announcer_packets_recv} / {announcer_packets_lost}    (send/recv/lost)

Skybit:
  Counters:
    ok         : {dlrs_skbt_ok}
    error      : {dlrs_skbt_err}
  Seeding:
    Memory:
      size     : {skbt_seed_mem_size[0]:>6} {skbt_seed_mem_size[1]}
      free     : {skbt_seed_mem_free[0]:>6} {skbt_seed_mem_free[1]}

Database (sqlite{sqlite_locked_str}):
  Memory info
    total      : {sqlite_mem_total[0]:>6} {sqlite_mem_total[1]}
    generic    : {sqlite_mem_generic[0]:>6} {sqlite_mem_generic[1]}
    cache      : {sqlite_mem_cache[0]:>6} {sqlite_mem_cache[1]}
    prep_stmt  : {sqlite_prep_stmt_used[0]:>6} {sqlite_prep_stmt_used[1]}
  Cache
    written    : {sqlite_cache_writes[0]:>6} {sqlite_cache_writes[1]}
    hits       : [{sqlite_cache_hit_percent:3.0%}] {sqlite_cache_hit:>{sqlite_cache_hit_miss_maxlen}}
    miss       : [{sqlite_cache_miss_percent:3.0%}] {sqlite_cache_miss:>{sqlite_cache_hit_miss_maxlen}}
  Data
    size       : {sqlite_data_used_bytes[0]:>6} {sqlite_data_used_bytes[1]}
    free       : {sqlite_data_free_bytes[0]:>6} {sqlite_data_free_bytes[1]} [{sqlite_data_free_percent:3.0%}]
  Maintanance
    vacuum     : {sqlite_maintainance_vacuum} ago (took {sqlite_maintainance_vacuum_duration})
    analyze    : {sqlite_maintainance_analyze} ago (took {sqlite_maintainance_analyze_duration})
  Locks
    Waiting    : {sqlite_job_waiters}
    Active
      meth     : {sqlite_active_job[0]}
      args     : {sqlite_active_job[1]}
      kwargs   : {sqlite_active_job[2]}

Proxy:
{proxy_status}

''', (), pstatus).strip())
    else:
        print(string.Formatter().vformat('''\
{status} | \
{uptime:>12} | \
{cpu_percent_self:>4.0%} | {cpu_percent_children:4.0%} \
|{line_job_stat}|| \
skbt {dlrs_skbt_ok:>5} {dlrs_skbt_err:>5} || \
look {proxy_infohash_time_spent:>8}
''', (), pstatus).strip())

    if status:
        print()
        print('Extra status items: %r' % (status, ))


def counters(args):
    all_counters = {
        'file_cache': {
            'data_size': 'Size of all data (excludes duplicate files)',
            'file_size': 'Size of all files (can be treated as size of all copier resources)',
            'resource_count': 'Count of resources',
            'file_count': 'Count of files'
        },
        'proxy': {
            'bytes_in': 'Amount of bytes received',
            'bytes_ou': 'Amonut of bytes sent',
            'bytes_in_torrent': 'Amount of bytes received via torrent',
            'bytes_ou_torrent': 'Amount of bytes sent via torrent',
            'bytes_in_skbt': 'Amount of bytes received via skybit',
            'bytes_ou_skbt': 'Amount of bytes sent via skybit',
            'connects_in': 'Amount of incoming connections',
            'connects_ou': 'Amount of connections made',
            'connects_in_torrent': 'Amount of incoming connections via torrent',
            'connects_ou_torrent': 'Amount of connections made via torrent',
            'connects_in_skbt': 'Amount of incoming connections via skybit',
            'connects_ou_skbt': 'Amount of connections made via skybit',
        },
        'rusage': {
            'uptime': 'Uptime in secs',
            'self_system': 'System usage in seconds',
            'chld_system': 'System usage of all childs in seconds',
            'self_user': 'User usage in seconds',
            'chld_user': 'User usage of all childs in seconds',
        },
    }

    if 'list' in args.key:
        for block, counters in all_counters.iteritems():
            print(block)
            for name, help in sorted(counters.iteritems()):
                print('  %-16s : %s' % (name, help))

        return

    query = []

    for key in args.key:
        if ':' in key:
            block, counters = key.split(':', 1)
            if block not in all_counters:
                print('Invalid block: %s' % (block, ))
                return 1

            counters = [c.strip() for c in counters.split(',')]

            for counter in counters:
                if counter not in all_counters[block]:
                    print('Invalid counter: %s:%s' % (block, counter))
                    return 1
        else:
            block = key
            if block not in all_counters:
                print('Invalid block: %s' % (block, ))
                return 1

            counters = all_counters[block].keys()

        query.append((block, counters))

    if query:
        try:
            result = retry_call('counters', query)
        except Exception as ex:
            print('Error: %s' % (ex, ))
            return 1
    else:
        result = {}

    if args.format != 'text':
        return _formatter(result, args.format)

    return result


def query(args):
    result = retry_call('query', args.sql)
    if result is None:
        print('result: NONE')
    else:
        if len(result) == 0:
            print('0 rows returned')
        else:
            cols = []
            resultset = []

            for row in result[:1000]:
                nrow = []
                for idx, col in enumerate(row):
                    value = str(col)
                    if len(cols) == idx:
                        cols.append(len(value))
                    else:
                        cols[idx] = max(cols[idx], len(value))

                    nrow.append(value)
                resultset.append(nrow)

            def header():
                return '+-' + '-+-'.join(['-' * cl for cl in cols]) + '-+'

            print(header())
            for row in resultset:
                frow = []
                for idx, col in enumerate(row):
                    collen = cols[idx]
                    frow.append('{0:<{len}}'.format(col, len=collen))

                print('| ' + ' | '.join(frow) + ' |')
            print(header())

            if len(result) > 1000:
                print('displayed first 1000 entries (%d total)' % (len(result), ))


def dbbackup(args):
    retry_call('dbbackup')


def dbcheck(args):
    ret = retry_call('dbcheck', full_resource_check=args.full_resource_check, fix=args.fix)
    print(ret)


def evaluate(args):
    print(retry_call('evaluate', args.code))


def _check(args, typ):
    if args.file == '-':
        fp = sys.stdin
    else:
        fp = open(args.file, 'rb')

    if args.msgpack:
        unpacker = msgpack.Unpacker()
        packer = msgpack.Packer()

        while True:
            data = fp.read(8192)
            if not data:
                break
            unpacker.feed(data)

        resources = list(unpacker)
    else:
        sep = '\0' if args.null else '\n'
        resources = [r for r in fp.read().split(sep) if r]

    if typ == 'resources':
        for idx, resid in enumerate(resources):
            if ':' in resid:
                resources[idx] = resid.split(':', 1)[1]

    max_block = 5000
    for i in range((len(resources) // max_block) + (1 if len(resources) % max_block else 0)):
        block = list(set(resources[i * max_block:(i + 1) * max_block]))
        meth = {
            'resources': lambda block: retry_call('check_resources', block),
            'paths': lambda block: retry_call('check_paths', block),
        }[typ]

        for failed in meth(block):
            if typ == 'resources':
                failed = 'rbtorrent:' + failed
            if args.msgpack:
                sys.stdout.write(packer.pack(failed))
            else:
                sys.stdout.write(failed + sep)


def check_resources(args):
    return _check(args, 'resources')


def check_paths(args):
    return _check(args, 'paths')


def notify(args):
    if not retry_call('notify', args.path):
        return 1


def file_move(args):
    if len(args.files) % 2 != 0:
        raise Exception('Got odd number of arguments (%d)' % (len(args.files), ))

    pairs = []
    has_errors = False

    for pair_idx in range(len(args.files) // 2):
        source = args.files[pair_idx * 2]
        target = args.files[pair_idx * 2 + 1]
        pairs.append((source, target))

    for src, (result, desc) in retry_call('file_move', pairs=pairs, after=args.after, quiet=args.quiet).items():
        if not result:
            print('%s: error: %s' % (src, desc))
            has_errors = True
        else:
            print('%s: done: %s' % (src, desc))

    if has_errors:
        return 1
    return 0


def main():
    # First of all, set locale to user's preferred.
    locale.setlocale(locale.LC_ALL, '')

    # Create the top-level parser.
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Copier maintenance tool.'
    )
    parser.add_argument(
        '-f', '--format',
        choices=['text', 'yaml', 'json', 'pprint', 'msgpack'],
        default='text',
        help='output format'
    )
    subparsers = parser.add_subparsers(help='operational mode')

    # Create 'status' sub-parser.
    subparser = subparsers.add_parser('status', help='show copier daemon status')
    subparser.add_argument('--line', action='store_true')
    subparser.add_argument('--wait', type=int, default=3)
    subparser.set_defaults(func=status)

    # Create 'counters' sub-parser
    subparser = subparsers.add_parser('counters', help='grab some counters from copier')
    subparser.add_argument('--cache', type=int, help='time in seconds for cache counters')
    subparser.add_argument('key', nargs='+', help='keys to query. To list all keys use "list"')
    subparser.set_defaults(func=counters)

    # Create 'query' sub-parser.
    subparser = subparsers.add_parser('query', help='query db')
    subparser.add_argument('sql', metavar='SQL', help='sql to execute')
    subparser.set_defaults(func=query)

    # Create 'dbbackup' sub-parser
    subparser = subparsers.add_parser('dbbackup', help='backup db')
    subparser.set_defaults(func=dbbackup)

    # Create 'dbcheck' sub-parser
    subparser = subparsers.add_parser('dbcheck', help='check db consistency')
    subparser.add_argument(
        '--full-resource-check', action='store_true', help='check resources by unpacking them (slow!)'
    )
    subparser.add_argument('--fix', action='store_true', help='fix all problems found (destructive!)')
    subparser.set_defaults(func=dbcheck)

    # Create 'eval' sub-parser
    subparser = subparsers.add_parser('eval', help='evaluate code')
    subparser.add_argument('code', metavar='CODE', help='code to evaluate')
    subparser.set_defaults(func=evaluate)

    # Create 'check-resources' sub-parser
    subparser = subparsers.add_parser('check-resources', help='check for resources exist in daemon')
    subparser.add_argument('file', metavar='FILE', help='file to read from (- for stdin)')
    subparser.add_argument('-0', dest='null', action='store_true', help='input is null-separated (output as well)')
    subparser.add_argument('--msgpack', action='store_true', help='stream is in msgpack')
    subparser.set_defaults(func=check_resources)

    # Create 'check-paths' sub-parser
    subparser = subparsers.add_parser('check-paths', help='check for files exist in daemon')
    subparser.add_argument('file', metavar='FILE', help='file to read from (- for stdin)')
    subparser.add_argument('-0', dest='null', action='store_true', help='input is null-separated (output as well)')
    subparser.add_argument('--msgpack', action='store_true', help='stream is in msgpack')
    subparser.set_defaults(func=check_paths)

    # Create 'notify' sub-parser
    subparser = subparsers.add_parser('notify', help='notify copier about changed/removed files')
    subparser.add_argument('path', metavar='PATH', nargs='+', help='paths which were changed/removed')
    subparser.set_defaults(func=notify)

    subparser = subparsers.add_parser('file-move', help='move one file using rename()')
    subparser.add_argument('-A', '--after', action='store_true', help='Dont move files, just record')
    subparser.add_argument('-q', '--quiet', action='store_true', help='Dont report success moves')
    subparser.add_argument('files', metavar='SRC TGT', nargs='+', help='source to move and target')
    subparser.set_defaults(func=file_move)

    # Create 'resource-info' sub-parser
    subparser = subparsers.add_parser('resource-info', help='resource info')
    subparser.add_argument(
        '--wait-trackers', type=int, help='wait all trackers specified amount of seconds'
    )
    subparser.add_argument('resid', help='resource id')
    subparser.set_defaults(func=resource_info)

    # Parse the args and call whatever function was selected.
    args = parser.parse_args()
    return args.func(args)


if __name__ == '__main__':
    raise SystemExit(main())
