from __future__ import absolute_import

import api.config
from api.copier import errors

from .transports.tgz import TransportFactoryTgz
from .transports.rbtorrent import TransportFactoryRBTorrent
from .transports.sandbox import TransportFactorySandBoxTask, TransportFactorySandBoxResource
from .transports.rsync import TransportFactoryRsync
from .transports.http import TransportFactoryHttp, TransportFactoryHttps
from .transports.generic import TransportFactoryAny, TransportFactoryAll


TRANSPORT_FACTORIES = (
    TransportFactoryTgz(),
    TransportFactoryRBTorrent(),
    TransportFactorySandBoxTask(),
    TransportFactorySandBoxResource(),
    TransportFactoryHttp(),
    TransportFactoryHttps(),
    TransportFactoryRsync(),
    TransportFactoryAll(),
    TransportFactoryAny(),
)


def find_transport(magic):
    for factory in TRANSPORT_FACTORIES:
        try:
            res = factory.check_magic(magic)
        except AttributeError:
            res = factory.checkMagic(magic)

        if res:
            try:
                ops = factory.ops()
            except AttributeError:
                ops = ()
            return factory.create(), ops

    raise errors.ApiError('Cant detect transport -- unknown transport?')


def detect_transport(resource_id):
    """
    Choose appropriate transport based on magic bytes
    """

    try:
        magic, resource_id = resource_id.split(':', 1)
    except ValueError:
        raise errors.ApiError('Cant detect transport -- invalid resource id format')

    return find_transport(magic), resource_id


class Handle(object):
    def __init__(self, slave, ops, skbn_sock):
        self.__sock = skbn_sock
        self.slave = slave
        self.ops = ops

    def list(self, priority=api.copier.Priority.Normal, network=api.copier.Network.Auto, **extra):
        kwargs = {}
        if 'priority' in self.ops:
            kwargs['priority'] = api.copier.Priority.toString(priority)
        if 'network' in self.ops:
            kwargs['network'] = api.copier.Network.toString(network)

        if 'api_sock' in self.ops:
            kwargs['api_sock'] = self.__sock

        kwargs.update(extra)

        return self.slave.list(**kwargs)

    def get(
        self, dest, user=False,
        priority='Normal', network='Auto',
        deduplicate='No',
        max_dl_speed=None, max_ul_speed=None,
        **extra
    ):
        kwargs = {'user': user}
        if 'priority' in self.ops:
            kwargs['priority'] = api.copier.Priority.toString(priority)
        if 'network' in self.ops:
            kwargs['network'] = api.copier.Network.toString(network)
        if 'deduplicate' in self.ops:
            kwargs['deduplicate'] = api.copier.Deduplicate.toString(deduplicate)
        if 'max_dl_speed' in self.ops:
            kwargs['max_dl_speed'] = max_dl_speed
        if 'max_ul_speed' in self.ops:
            kwargs['max_ul_speed'] = max_ul_speed

        if 'kw' in self.ops:
            kwargs.update(extra)

        if 'api_sock' in self.ops:
            kwargs['api_sock'] = self.__sock

        return self.slave.get(dest, **kwargs)


class Copier(object):
    def __init__(self, sock=None):
        self.__sock = sock

    def create(self, files, transport=None, **extra):
        """
        This method creates resource from 'files' (recursively for directories)
        'transport' - desired transmission method. If None then system will choose.

        Returns object that has methods:
         wait() - wait for createEx() to complete.
        """

        if not transport:
            transport = api.config.copierTransport()

        transport, ops = find_transport(transport)

        if 'api_sock' in ops:
            extra['api_sock'] = self.__sock

        if 'kw' in ops:
            return transport.create(files, **extra)

        return transport.create(files)

    def handle(self, resid):
        (transport, ops), resource_id = detect_transport(resid)
        return Handle(transport.handle(resource_id), ops, skbn_sock=self.__sock)

    def copy(self, srcHost, srcPath, tgtHost, tgtPath, priority, srcUser, tgtUser):  # noqa
        from .simplecopy import simplecopy
        return simplecopy.copy(
            srcHost, srcPath, tgtHost, tgtPath,
            priority,
            srcUser, tgtUser
        )

    @staticmethod
    def transports_info():
        """
        This method is not in the public API do not use it outside skynet's code.
        It's only purpose is to provide information about available transports to the sky command.

        Returns information about available transports, list of tuples with two elements:
          - transports' magics
          - transports' docstrings
        """
        res = []
        for t in TRANSPORT_FACTORIES:
            try:
                try:
                    magic = t.get_magic()
                except AttributeError:
                    magic = t.getMagic()
            except Exception as ex:
                import warnings
                warnings.warn('Failed to grab transport info for %r: %s' % (
                    t, str(ex)
                ))
                continue

            try:
                res.append((magic, t.create().__doc__ or ''))
            except Exception as ex:
                res.append((magic, '(failed with: "%s")' % (str(ex), )))
        return res

    transportsInfo = transports_info
