from kernel.util.sys.dirut import ensureDirs

import subprocess
import os


def ensure_writable_dirs(target):
    prev = os.umask(0)
    try:
        ensureDirs(target)
    except OSError:
        raise OSError("Cannot create dir: %s" % os.path.dirname(target))
    finally:
        os.umask(prev)


class TaskScp(object):
    devnull = None

    def __init__(self, params, chmod_dest=False, user=None):
        super(TaskScp, self).__init__()
        self.params = params
        self.params.insert(0, '-BCp')
        if user is not None:
            self.params.insert(0, 'User={}'.format(user))
            self.params.insert(0, '-o')
        self.params.insert(0, 'scp')
        self.chmod_dest = chmod_dest
        self.proc = None
        self.returncode = None

    def start(self):
        if TaskScp.devnull is None:
            TaskScp.devnull = open('/dev/null', 'wb')
        self.proc = subprocess.Popen(self.params,
                                     shell=False,
                                     close_fds=True,
                                     stdout=TaskScp.devnull,
                                     stderr=TaskScp.devnull
                                     )

    def finished(self):
        if self.returncode is None and self.proc is not None:
            self.proc.poll()
            self.returncode = self.proc.returncode
        return self.returncode is not None

    @property
    def reason(self):
        if self.returncode is not None and self.returncode != 0:
            return RuntimeError("[scp error {}]".format(self.returncode))

    @property
    def product(self):
        return self.returncode

    def wait(self):
        if self.returncode is None:
            self.proc.communicate()
            self.returncode = self.proc.returncode
            self.proc = None

        if self.returncode != 0:
            return RuntimeError("[scp error {}]".format(self.returncode))
        return 0


class Proxy(object):
    def __init__(self, obj):
        self.obj = obj
        self.result = None
        self.copyId = None

    def start(self):
        self.obj.start()

    def wait(self):
        if self.result is None:
            self.result = self.obj.wait()
        return self.result

    def __getattr__(self, name):
        return getattr(self.obj, name)


def _copy(params, id, chmod_dest=False, user=None):
    task = TaskScp(params, chmod_dest=chmod_dest, user=user)

    task = Proxy(task)
    task.copyId = id
    task.start()
    return task


def _prepare_for_download(source, dest):
    if dest.endswith('/'):
        dest = os.path.join(dest, os.path.basename(source))
    dest = os.path.realpath(dest)
    if not os.path.isdir(dest):
        if not os.path.isdir(os.path.dirname(dest)):
            ensure_writable_dirs(os.path.dirname(dest))
    else:
        dest = os.path.join(dest, os.path.basename(source))


def _format_host_prefix(host, user):
    loopbacks = ["localhost", "127.0.0.1", "::1"]
    if user is not None:
        return '{}@{}:'.format(user, host)

    if host not in loopbacks:
        return '{}:'.format(host)

    return ''


def copy(sourceHost, source, targetHost, target, priority, srcUser, tgtUser):  # noqa
    id = "%s:%s -> %s:%s" % (sourceHost, source, targetHost, target)

    source_host = _format_host_prefix(sourceHost, srcUser)
    target_host = _format_host_prefix(targetHost, tgtUser)
    chmod_dest = False

    if not targetHost:
        _prepare_for_download(source, target)
        chmod_dest = True

    return _copy([
        '{}{}'.format(source_host, source),
        '{}{}'.format(target_host, target),
    ], id, chmod_dest=chmod_dest)
