# coding: utf-8

"""
Generic transports.

Transports implemented here do it's job by calling low-level transports in
various combinations.
"""

from __future__ import absolute_import

import itertools
import collections

import api.copier
from api.copier.errors import ResourceError

from kernel.util.functional import singleton

from ...rbtorrent import logger as logging
from . import BaseTransport


@singleton
def logger():
    return logging.initialize_client().getChild('generic')


class HandlerBase(object):
    def __init__(self, resid, downloader_maker=None, downloader_waiter=None):
        self._resid = resid
        self._downloader = None
        self._downloader_maker = downloader_maker
        self._downloader_waiter = downloader_waiter

        self._result = None
        self._result_ready = False

    def resid(self):
        return self._resid

    def iter(self, timeout=None, state_version=1):
        result = self._downloader_waiter(self._downloader, timeout)
        self._result = result
        self._result_ready = True

        return
        yield result  # forcing this func to be a generator

    def wait(self, timeout=None):
        if self._result_ready:
            return self._result

        for _ in self.iter(timeout=timeout):
            pass

        assert self._result_ready
        return self._result


class Getter(HandlerBase):
    def __init__(self, resid):
        super(Getter, self).__init__(
            resid,
            (
                lambda
                copier, resource, dest, user,
                priority, network, deduplicate,
                max_dl_speed, max_ul_speed:
                copier.get(
                    resource, dest, user, priority, network
                )
            ),
            lambda downloader, timeout: downloader.wait(timeout)
        )


class Lister(HandlerBase):
    def __init__(self, resid):
        super(Lister, self).__init__(
            resid,
            lambda copier, resource, dest, user,
            priority, network, deduplicate,
            max_dl_speed, max_ul_speed:
            copier.handle(resource).list(
                priority=priority,
                network=network
            ),
            lambda downloader, timeout: downloader.wait(timeout).files()
        )


class TransportFactoryBase(object):
    def __init__(self, magic, transport):
        self.__magic = magic
        self.__transport = transport

    def getMagic(self):  # noqa
        return self.__magic

    def checkMagic(self, magic):  # noqa
        return magic in self.getMagic()

    def create(self):
        return self.__transport()

    def ops(self):
        return 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed'


class TransportGenericError(ResourceError):
    """
    Generic transport error.
    """
    pass


class BaseAll(HandlerBase):
    def __init__(self, resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        super(BaseAll, self).__init__(resid)

        c = api.copier.Copier()
        self.__downloaders = []
        log = logger().getChild('all')
        for resid in self._resid.split(','):
            log.debug('Obtaining the resource %r', resid)
            self.__downloaders.append(
                self._downloader_maker(
                    c, resid, dest, user,
                    priority, network, deduplicate,
                    max_dl_speed, max_ul_speed,
                )
            )
        pass

    def iter(self, timeout=None, state_version=1):
        if not timeout:
            timeout = None
        ret = []
        for d in self.__downloaders:
            ret.append(self._downloader_waiter(d, timeout))
        self._result = ret
        self._result_ready = True
        return
        yield


class GetAll(BaseAll, Getter):
    """
    Fetches all the resources specified in parallel.
    If some of resources fetch operation will fail, an exception will be raised.
    """
    pass


class ListAll(BaseAll, Lister):
    """
    List all the resources specified in parallel.
    If some of resources list operation will fail, an exception will be raised.
    """
    def iter(self, timeout=None, state_version=1):
        for x in super(ListAll, self).iter(timeout):
            pass

        assert self._result_ready

        self._result = sorted(
            itertools.chain.from_iterable(
                self._result
            ),
            key=lambda x: x['name']
        )

        return
        yield


class BaseAny(HandlerBase):
    def __init__(self, resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        super(BaseAny, self).__init__(resid)

        c = api.copier.Copier()
        log = logger().getChild('any')

        self._downloader_maker = lambda resource, mk=self._downloader_maker: \
            mk(c, resource, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed)
        self.__downloaders = collections.OrderedDict.fromkeys(resid.split(','))
        for resid in self.__downloaders.iterkeys():
            try:
                log.debug('Trying the resource %r', resid)
                self.__downloaders[resid] = self._downloader_maker(resid)
                return
            except Exception as e:
                self.__downloaders[resid] = e
                log.warn('Resource %r get failed: %s', resid, str(e))

        # Unable to fetch any of resources specified. Compile a final message
        raise TransportGenericError(self._cumulative_error_message())

    def iter(self, timeout=None, state_version=1):
        if not timeout:
            timeout = None

        log = logger().getChild('any')
        for resid, downloader in self.__downloaders.items():
            try:
                if downloader is None:
                    log.debug('Trying the resource %r', resid)
                    downloader = self._downloader_maker(resid)
                elif isinstance(downloader, Exception):
                    continue
                self._result = self._downloader_waiter(downloader, timeout)
                self._result_ready = True
                return
            except Exception as e:
                self.__downloaders[resid] = e
                log.warn('Resource %r get failed: %s', resid, str(e))

        # Unable to fetch any of resources specified. Compile a final message
        raise TransportGenericError(self._cumulative_error_message())

        yield  # force this meth to be a generator

    def _cumulative_error_message(self):
        """
        Compile an cumulative error message about all the errors are
        collected while the transport tries to download resources specified.
        """
        return "\n".join(
            ['Can\'t fetch any resource:'] +
            [
                'unable to get %r: %s' % (resid, ex)
                for resid, ex in self.__downloaders.iteritems()
            ]
        )


class GetAny(BaseAny, Getter):
    """
    Tries to fetch any of the resources specified one-by-one.
    If some of resources fetch operation will fail, an error will be shown.
    In case of none of resources fetched, an exception will be generated.
    """
    pass


class ListAny(BaseAny, Lister):
    """
    Tries to list any of the resources specified one-by-one.
    If some of resources list operation will fail, an error will be shown.
    In case of none of resources listed, an exception will be generated.
    """
    pass


class GenericHandle(HandlerBase):
    def __init__(self, resid, download_all):
        super(GenericHandle, self).__init__(resid)
        self.__all = download_all

    def get(self, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        return (GetAll if self.__all else GetAny)(
            self._resid, dest, user,
            priority, network, deduplicate,
            max_dl_speed, max_ul_speed,
        )

    def list(self, priority, network):
        return (ListAll if self.__all else ListAny)(
            self._resid, None, None,
            priority, network, None,
            None, None,
        )


class TransportAll(BaseTransport):
    """
    All Transport. Downloads __all__ of resources specified.
    As <resid> accepts any valid <resid>'s separated by commma.
    """
    def handle(self, resid):
        return GenericHandle(resid, True)

    def create(self, *args, **kw):
        raise TransportGenericError("All and Any transports do not supports resource creation")


class TransportAny(BaseTransport):
    """
    Any Transport. Downloads __any__ of resources specified.
    As <resid> accepts any valid <resid>'s separated by commma.
    """
    def handle(self, resid):
        return GenericHandle(resid, False)

    def create(self, *args, **kw):
        raise TransportGenericError("All and Any transports do not supports resource creation")


# Factories
class TransportFactoryAny(TransportFactoryBase):
    """
    Factory for any transports
    """
    def __init__(self):
        super(TransportFactoryAny, self).__init__(['any'], TransportAny)

    def ops(self):
        return 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed'


class TransportFactoryAll(TransportFactoryBase):
    """
    Factory for all transports
    """
    def __init__(self):
        super(TransportFactoryAll, self).__init__(['all'], TransportAll)

    def ops(self):
        return 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed'
