# coding: utf-8
from __future__ import absolute_import

import multiprocessing
import os
import sys
import stat
import urlparse

import requests

from api.copier.errors import ResourceError

from . import BaseTransport


class TransportHttpError(ResourceError):
    '''
    Http transport error.
    '''
    pass


def _download(url, dest):
    default_open_flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
    if sys.platform == 'win32':
        default_open_flags |= os.O_BINARY
    default_open_mode = stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO

    raw_download = requests.get(url, verify=False, stream=True).raw
    with os.fdopen(os.open(dest, default_open_flags, default_open_mode), 'wb') as fp:
        while True:
            chunk = raw_download.read(8096)
            if not chunk:
                break
            fp.write(chunk)
    pass


class HttpGet(object):
    '''
    XXX: This transport doesn't support async download.
    wait must always be called to correct finalization.
    '''
    def __init__(self, resid, dest, network):
        try:
            from ya.skynet.services.copier.client.utils import fastbonizeURL
        except ImportError:
            from ya.skynet.services.skybone.client.utils import fastbonizeURL

        self.__resid = resid
        self.__dest = dest

        self._result = None
        self._result_ready = False

        if network in ('Auto', 'Fastbone'):
            url = fastbonizeURL(resid)
        else:
            url = resid

        if os.path.isdir(self.__dest):
            fname = urlparse.urlparse(url).path.split('/')[-1]
            if not fname:
                raise TransportHttpError(
                    "Can't create destination path from url. Use -d option")
            dst = os.path.join(self.__dest, fname)
        else:
            dst = self.__dest

        # start download in a subprocess
        try:
            self.downloader = multiprocessing.Process(target=_download, args=(url, dst))
            self.downloader.start()
        except Exception as e:
            raise TransportHttpError("Error while downloading resource: %s" % e)

    def resid(self):
        return self.__resid

    def iter(self, timeout=None, state_version=1):
        # this transport doesn't work in async mode
        if timeout == 0:
            self.downloader.terminate()
            raise TransportHttpError("Async download is not supported")
        self.downloader.join(timeout)
        if self.downloader.is_alive():
            self.downloader.terminate()
            raise TransportHttpError("Timeout %s exceeded" % timeout)
        if self.downloader.exitcode > 0:
            raise TransportHttpError(
                "Error while downloading resource: process terminated with %s" % self.downloader.exitcode)
        elif self.downloader.exitcode < 0:
            raise TransportHttpError(
                "Error while downloading resource: process kill by signal: %s" % -self.downloader.exitcode)

        self._result_ready = True
        return
        yield

    def wait(self, timeout=None):
        if self._result_ready:
            return self._result

        for _ in self.iter(timeout=timeout):
            pass

        assert self._result_ready
        return self._result


class HttpHandle(object):
    def __init__(self, resid):
        self.__resid = resid

    def resid(self):
        return self.__resid

    def get(self, dest, user=False, network='Auto'):
        try:
            from ya.skynet.services.copier.client.utils import umask
        except ImportError:
            from ya.skynet.services.skybone.client.utils import umask

        with umask(None if user else 0000):
            return HttpGet(self.__resid, dest, network)

    def list(self):
        raise TransportHttpError("Http transport does not supports listing")


class TransportHttp(BaseTransport):
    '''
    HTTP Transport.
    As <resid> accepts http(s) url.
    Note: this transport works only in synchronous mode.
    '''
    def __init__(self, ssl):
        self.__ssl = ssl

    def _check_and_update_resid(self, resid):
        if resid.startswith('//'):
            return ('https:' if self.__ssl else 'http:') + resid
        elif not resid.startswith(('http://', 'https://')):
            raise TransportHttpError("Bad resid: '%s'" % resid)
        return resid

    def handle(self, resid):
        return HttpHandle(self._check_and_update_resid(resid))

    def create(self, *args, **kw):
        raise TransportHttpError("Http transport does not supports resource creation")


class TransportFactoryHttp(object):
    """
    Factory for http transports
    """
    def getMagic(self):   # noqa
        return ['http']

    def checkMagic(self, magic):  # noqa
        return magic in self.getMagic()

    def create(self):
        return TransportHttp(ssl=False)

    def ops(self):
        return 'network',


class TransportFactoryHttps(object):
    """
    Factory for http transports
    """
    def getMagic(self):  # noqa
        return ['https']

    def checkMagic(self, magic):  # noqa
        return magic in self.getMagic()

    def create(self):
        return TransportHttp(ssl=True)

    def ops(self):
        return 'network',
